#!/usr/bin/env python3
"""
Build script to generate HDRIProcessor.gizmo with embedded SphericalBlur kernel.

This script reads the SphericalBlur.blink file and embeds it into the gizmo,
ensuring the kernel is always loaded correctly.

Usage:
    python scripts/build_hdri_processor.py

Run this whenever you modify SphericalBlur.blink to update the gizmo.
"""

import os
import re


def escape_for_gizmo(kernel_source: str) -> str:
    """Escape kernel source for embedding in a Nuke gizmo file."""
    # Escape backslashes first
    escaped = kernel_source.replace("\\", "\\\\")
    # Escape curly braces
    escaped = escaped.replace("{", "\\{")
    escaped = escaped.replace("}", "\\}")
    # Escape quotes
    escaped = escaped.replace('"', '\\"')
    # Replace newlines with \n
    escaped = escaped.replace("\n", "\\n")
    return escaped


def generate_blinkscript_node(
    name: str,
    label: str,
    kernel_source_escaped: str,
    xpos: int,
    ypos: int,
) -> str:
    """Generate a BlinkScript node definition for the gizmo."""
    return f''' BlinkScript {{
  recompileCount 1
  ProgramGroup 1
  kernelSource "{kernel_source_escaped}"
  rebuild ""
  rebuild_finalise ""
  name {name}
  label "{label}"
  xpos {xpos}
  ypos {ypos}
 }}'''


def main():
    # Get paths
    script_dir = os.path.dirname(os.path.abspath(__file__))
    addon_dir = os.path.dirname(script_dir)
    blink_path = os.path.join(addon_dir, "blinkscripts", "SphericalBlur.blink")
    gizmo_path = os.path.join(addon_dir, "gizmos", "HDRIProcessor.gizmo")

    # Read the SphericalBlur.blink file
    print(f"Reading: {blink_path}")
    with open(blink_path, "r") as f:
        kernel_source = f.read()

    # Escape for embedding in gizmo
    kernel_escaped = escape_for_gizmo(kernel_source)
    print(f"Kernel source length: {len(kernel_source)} chars")

    # Generate the gizmo content
    gizmo_content = f"""#! /Applications/Nuke14.1v5/Nuke14.1v5.app/Contents/MacOS/libnuke-14.1.5.dylib -nx
version 14.1 v2
Gizmo {{
 inputs 1
 label "Beeble HDRI Processor"
 addUserKnob {{20 User l "HDRI Settings"}}
 addUserKnob {{26 info l "" +STARTLINE T "Spherical Gaussian blur for seamless IBL.\\nTwo-pass separable blur (latitudinal + longitudinal).\\n\\nInput is resized to max 512px (aspect preserved).\\n\\nOutput layers:\\n- specular_0: roughness 0.0 (no blur)\\n- specular_1: roughness 0.2 (sigma 0.125)\\n- specular_2: roughness 0.4 (sigma 0.215)\\n- specular_3: roughness 0.6 (sigma 0.35)\\n- specular_4: roughness 0.8 (sigma 0.446)\\n- specular_5: roughness 1.0 (sigma 0.582)\\n- diffuse: hemisphere blur (sigma 1.5)\\n\\nBased on Three.js PMREMGenerator."}}
}}
 BackdropNode {{
  inputs 0
  name BackdropNode1
  label "Beeble HDRI Processor\\nSpherical Blur (Two-Pass)"
  xpos 0
  ypos 50
  bdwidth 1200
  bdheight 750
 }}
 Input {{
  inputs 0
  name HDRI
  label "HDRI Input"
  xpos 500
  ypos 80
 }}
 Reformat {{
  type "to box"
  box_width 512
  box_height 512
  box_fixed true
  resize fit
  filter Cubic
  pbb true
  name ResizeToMax512
  label "Resize to max 512\\n(aspect preserved)"
  xpos 500
  ypos 120
 }}
 Dot {{
  name Dot1
  xpos 534
  ypos 170
 }}
set NHDRI [stack 0]
{generate_blinkscript_node("DiffuseBlurLat", "Diffuse Pass 1\\n(latitudinal)", kernel_escaped, 50, 220)}
{generate_blinkscript_node("DiffuseBlurLon", "Diffuse Pass 2\\n(longitudinal)", kernel_escaped, 50, 280)}
add_layer {{diffuse diffuse.red diffuse.green diffuse.blue}}
 Shuffle2 {{
  fromInput1 {{{{0}} B}}
  in1 rgba
  out1 diffuse
  fromInput2 {{{{0}} B}}
  mappings "4 rgba.red 0 0 diffuse.red 0 0 rgba.green 0 1 diffuse.green 0 1 rgba.blue 0 2 diffuse.blue 0 2 rgba.alpha 0 3 diffuse.alpha 0 3"
  name CopyToDiffuse
  xpos 50
  ypos 340
 }}
push $NHDRI
add_layer {{specular_0 specular_0.red specular_0.green specular_0.blue}}
 Shuffle2 {{
  fromInput1 {{{{0}} B}}
  in1 rgba
  out1 specular_0
  fromInput2 {{{{0}} B}}
  mappings "4 rgba.red 0 0 specular_0.red 0 0 rgba.green 0 1 specular_0.green 0 1 rgba.blue 0 2 specular_0.blue 0 2 rgba.alpha 0 3 specular_0.alpha 0 3"
  name CopyToSpecular0
  label "specular_0\\n(roughness 0.0)\\nno blur"
  xpos 180
  ypos 220
 }}
push $NHDRI
{generate_blinkscript_node("Spec1Lat", "roughness 0.2\\nPass 1 (lat)", kernel_escaped, 310, 220)}
{generate_blinkscript_node("Spec1Lon", "roughness 0.2\\nPass 2 (lon)", kernel_escaped, 310, 280)}
add_layer {{specular_1 specular_1.red specular_1.green specular_1.blue}}
 Shuffle2 {{
  fromInput1 {{{{0}} B}}
  in1 rgba
  out1 specular_1
  fromInput2 {{{{0}} B}}
  mappings "4 rgba.red 0 0 specular_1.red 0 0 rgba.green 0 1 specular_1.green 0 1 rgba.blue 0 2 specular_1.blue 0 2 rgba.alpha 0 3 specular_1.alpha 0 3"
  name CopyToSpecular1
  label "specular_1\\n(roughness 0.2)"
  xpos 310
  ypos 340
 }}
push $NHDRI
{generate_blinkscript_node("Spec2Lat", "roughness 0.4\\nPass 1 (lat)", kernel_escaped, 440, 220)}
{generate_blinkscript_node("Spec2Lon", "roughness 0.4\\nPass 2 (lon)", kernel_escaped, 440, 280)}
add_layer {{specular_2 specular_2.red specular_2.green specular_2.blue}}
 Shuffle2 {{
  fromInput1 {{{{0}} B}}
  in1 rgba
  out1 specular_2
  fromInput2 {{{{0}} B}}
  mappings "4 rgba.red 0 0 specular_2.red 0 0 rgba.green 0 1 specular_2.green 0 1 rgba.blue 0 2 specular_2.blue 0 2 rgba.alpha 0 3 specular_2.alpha 0 3"
  name CopyToSpecular2
  label "specular_2\\n(roughness 0.4)"
  xpos 440
  ypos 340
 }}
push $NHDRI
{generate_blinkscript_node("Spec3Lat", "roughness 0.6\\nPass 1 (lat)", kernel_escaped, 570, 220)}
{generate_blinkscript_node("Spec3Lon", "roughness 0.6\\nPass 2 (lon)", kernel_escaped, 570, 280)}
add_layer {{specular_3 specular_3.red specular_3.green specular_3.blue}}
 Shuffle2 {{
  fromInput1 {{{{0}} B}}
  in1 rgba
  out1 specular_3
  fromInput2 {{{{0}} B}}
  mappings "4 rgba.red 0 0 specular_3.red 0 0 rgba.green 0 1 specular_3.green 0 1 rgba.blue 0 2 specular_3.blue 0 2 rgba.alpha 0 3 specular_3.alpha 0 3"
  name CopyToSpecular3
  label "specular_3\\n(roughness 0.6)"
  xpos 570
  ypos 340
 }}
push $NHDRI
{generate_blinkscript_node("Spec4Lat", "roughness 0.8\\nPass 1 (lat)", kernel_escaped, 700, 220)}
{generate_blinkscript_node("Spec4Lon", "roughness 0.8\\nPass 2 (lon)", kernel_escaped, 700, 280)}
add_layer {{specular_4 specular_4.red specular_4.green specular_4.blue}}
 Shuffle2 {{
  fromInput1 {{{{0}} B}}
  in1 rgba
  out1 specular_4
  fromInput2 {{{{0}} B}}
  mappings "4 rgba.red 0 0 specular_4.red 0 0 rgba.green 0 1 specular_4.green 0 1 rgba.blue 0 2 specular_4.blue 0 2 rgba.alpha 0 3 specular_4.alpha 0 3"
  name CopyToSpecular4
  label "specular_4\\n(roughness 0.8)"
  xpos 700
  ypos 340
 }}
push $NHDRI
{generate_blinkscript_node("Spec5Lat", "roughness 1.0\\nPass 1 (lat)", kernel_escaped, 830, 220)}
{generate_blinkscript_node("Spec5Lon", "roughness 1.0\\nPass 2 (lon)", kernel_escaped, 830, 280)}
add_layer {{specular_5 specular_5.red specular_5.green specular_5.blue}}
 Shuffle2 {{
  fromInput1 {{{{0}} B}}
  in1 rgba
  out1 specular_5
  fromInput2 {{{{0}} B}}
  mappings "4 rgba.red 0 0 specular_5.red 0 0 rgba.green 0 1 specular_5.green 0 1 rgba.blue 0 2 specular_5.blue 0 2 rgba.alpha 0 3 specular_5.alpha 0 3"
  name CopyToSpecular5
  label "specular_5\\n(roughness 1.0)"
  xpos 830
  ypos 340
 }}
 Merge2 {{
  inputs 2
  operation copy
  also_merge all
  name Merge1
  xpos 700
  ypos 400
 }}
 Merge2 {{
  inputs 2
  operation copy
  also_merge all
  name Merge2
  xpos 570
  ypos 440
 }}
 Merge2 {{
  inputs 2
  operation copy
  also_merge all
  name Merge3
  xpos 440
  ypos 480
 }}
 Merge2 {{
  inputs 2
  operation copy
  also_merge all
  name Merge4
  xpos 310
  ypos 520
 }}
 Merge2 {{
  inputs 2
  operation copy
  also_merge all
  name Merge5
  xpos 180
  ypos 560
 }}
 Merge2 {{
  inputs 2
  operation copy
  also_merge all
  name MergeDiffuse
  xpos 50
  ypos 600
 }}
 Output {{
  name Output1
  xpos 50
  ypos 660
 }}
end_group
"""

    # Write the gizmo file
    print(f"Writing: {gizmo_path}")
    with open(gizmo_path, "w") as f:
        f.write(gizmo_content)

    print("Done! HDRIProcessor.gizmo has been updated with embedded kernel source.")
    print(
        "\nRemember to update lighting.py to set the parameter values after node creation."
    )


if __name__ == "__main__":
    main()
