# beeble/update_checker.py
"""Update checker for Beeble Nuke plugin with auto-install support."""

import json
import os
import shutil
import ssl
import subprocess
import sys
import tempfile
import zipfile

import nuke

try:
    from urllib.request import urlopen, Request
except ImportError:
    from urllib2 import urlopen, Request

UPDATE_URL = "https://release.beeble.ai/plugins/nuke/latest.json"
DOWNLOAD_URL = "https://release.beeble.ai/plugins/nuke/latest.zip"


# Create SSL context that doesn't verify certificates (for Nuke compatibility)
def _create_ssl_context():
    """Create an SSL context that bypasses certificate verification."""
    ctx = ssl.create_default_context()
    ctx.check_hostname = False
    ctx.verify_mode = ssl.CERT_NONE
    return ctx


def _parse_version(version_str):
    """Parse version string into comparable tuple."""
    # Handle versions like "0.5.0", "1.2.3", etc.
    parts = version_str.strip().split(".")
    result = []
    for part in parts:
        # Extract numeric portion (handles cases like "1v5" -> 1)
        num = ""
        for char in part:
            if char.isdigit():
                num += char
            else:
                break
        result.append(int(num) if num else 0)
    return tuple(result)


def _fetch_latest_version():
    """Fetch latest version info from server. Returns dict or None on error."""
    try:
        req = Request(UPDATE_URL, headers={"User-Agent": "Beeble-Nuke-Plugin"})
        # Try with SSL context, fall back without if not supported
        try:
            ssl_context = _create_ssl_context()
            response = urlopen(req, timeout=10, context=ssl_context)
        except TypeError:
            # Python version doesn't support context parameter
            response = urlopen(req, timeout=10)
        return json.loads(response.read().decode("utf-8"))
    except Exception:
        return None


def _get_plugin_dir():
    """Get the root plugin directory (parent of beeble package)."""
    return os.path.dirname(os.path.dirname(os.path.abspath(__file__)))


def _download_and_stage_update():
    """Download update zip and extract to staging folder.

    Returns:
        tuple: (success: bool, error_message: str or None)
    """
    plugin_dir = _get_plugin_dir()
    staging_dir = os.path.join(plugin_dir, "_update_staging")

    try:
        # Clean up any existing staging directory
        if os.path.exists(staging_dir):
            shutil.rmtree(staging_dir)

        # Download zip to temp file
        print("[Beeble] Downloading update...")
        req = Request(DOWNLOAD_URL, headers={"User-Agent": "Beeble-Nuke-Plugin"})
        try:
            ssl_context = _create_ssl_context()
            response = urlopen(req, timeout=60, context=ssl_context)
        except TypeError:
            response = urlopen(req, timeout=60)

        # Save to temp file
        with tempfile.NamedTemporaryFile(delete=False, suffix=".zip") as tmp_file:
            tmp_path = tmp_file.name
            tmp_file.write(response.read())

        print("[Beeble] Extracting update...")
        # Extract to staging directory
        os.makedirs(staging_dir, exist_ok=True)
        with zipfile.ZipFile(tmp_path, "r") as zip_ref:
            # Check if zip has a root folder or contents directly
            names = zip_ref.namelist()
            # Find common prefix (root folder in zip)
            if names:
                first_part = names[0].split("/")[0]
                has_root_folder = all(
                    n.startswith(first_part + "/") or n == first_part for n in names
                )
            else:
                has_root_folder = False

            if has_root_folder:
                # Extract to temp, then move contents up
                temp_extract = os.path.join(staging_dir, "_temp_extract")
                zip_ref.extractall(temp_extract)
                root_folder = os.path.join(temp_extract, first_part)
                for item in os.listdir(root_folder):
                    src = os.path.join(root_folder, item)
                    dst = os.path.join(staging_dir, item)
                    shutil.move(src, dst)
                shutil.rmtree(temp_extract)
            else:
                zip_ref.extractall(staging_dir)

        # Clean up temp file
        os.remove(tmp_path)
        print("[Beeble] Update staged successfully!")

        return True, None

    except Exception as e:
        # Clean up on failure
        if os.path.exists(staging_dir):
            shutil.rmtree(staging_dir)
        return False, str(e)


def _get_nuke_executable():
    """Get the Nuke executable path for the currently running Nuke version."""
    # Use nuke.EXE_PATH - this is the most reliable way to get the current Nuke executable
    # It returns the exact path to the Nuke that's currently running
    try:
        exe_path = nuke.EXE_PATH
        if exe_path and os.path.exists(exe_path):
            return exe_path
    except AttributeError:
        pass

    # Fallback: Try environment variable
    if "NUKE_EXE" in os.environ:
        return os.environ["NUKE_EXE"]

    return None


def _restart_nuke():
    """Restart Nuke by launching new instance and exiting current."""
    nuke_exe = _get_nuke_executable()

    if nuke_exe and os.path.exists(nuke_exe):
        try:
            # Launch new Nuke instance
            if sys.platform == "win32":
                subprocess.Popen([nuke_exe], shell=False)
            else:
                subprocess.Popen([nuke_exe], start_new_session=True)

            # Exit current Nuke
            nuke.scriptExit()
        except Exception as e:
            nuke.message(
                "Failed to restart Nuke: {}\n\nPlease restart Nuke manually.".format(e)
            )
    else:
        nuke.message(
            "Update staged successfully!\n\n"
            "Could not auto-restart Nuke.\n"
            "Please restart Nuke manually to complete the update."
        )


def _perform_update():
    """Download, stage, and restart to apply update."""
    # Download happens without blocking dialog (progress shown in console)
    success, error = _download_and_stage_update()

    if success:
        nuke.message(
            "Update downloaded successfully!\n\n"
            "Nuke will now restart to apply the update."
        )
        _restart_nuke()
    else:
        nuke.message("Failed to download update:\n\n{}".format(error))


def check_for_updates():
    """Check for plugin updates and show dialog (always shows result)."""
    from . import __version__

    data = _fetch_latest_version()
    if data is None:
        nuke.message(
            "Failed to check for updates.\nPlease check your internet connection."
        )
        return

    latest = data.get("version")
    if _parse_version(latest) > _parse_version(__version__):
        # Ask user if they want to update
        message = (
            "Update available!\n\n"
            "Current: {}\n"
            "Latest: {}\n\n"
            "{}\n\n"
            "This will require restarting Nuke.\n"
            "Do you want to update now?"
        ).format(__version__, latest, data.get("message", ""))

        if nuke.ask(message):
            _perform_update()
    else:
        nuke.message("You're up to date!\n\nInstalled version: {}".format(__version__))


def check_for_updates_on_startup():
    """Silent update check on startup - asks user if update is available."""
    from . import __version__

    data = _fetch_latest_version()
    if data is None:
        return  # Silently fail on startup

    latest = data.get("version")
    if _parse_version(latest) > _parse_version(__version__):
        # Ask user if they want to update
        message = (
            "Beeble Plugin Update Available!\n\n"
            "Current: {}\n"
            "Latest: {}\n\n"
            "{}\n\n"
            "This will require restarting Nuke.\n"
            "Do you want to update now?"
        ).format(__version__, latest, data.get("message", ""))

        if nuke.ask(message):
            _perform_update()
