# beeble/loaders.py
"""Functions for loading PBR passes."""

import nuke
import os
import re


def add_pbr_controller():
    """
    Creates a PBRController gizmo node in the node graph.
    """
    # Get the path to the PBRController gizmo
    package_dir = os.path.dirname(os.path.dirname(__file__))
    gizmo_path = os.path.join(package_dir, "gizmos", "PBRController.gizmo")
    gizmo_path = gizmo_path.replace("\\", "/")

    # Create the gizmo
    pbr_controller = nuke.createNode(gizmo_path)
    return pbr_controller


def add_pbr_packer():
    """
    Creates a PBRPacker gizmo node in the node graph.
    """
    package_dir = os.path.dirname(os.path.dirname(__file__))
    gizmo_path = os.path.join(package_dir, "gizmos", "PBRPacker.gizmo")
    gizmo_path = gizmo_path.replace("\\", "/")

    pbr_packer = nuke.createNode(gizmo_path)
    return pbr_packer


# PBR pass configuration
PBR_PASSES = [
    "Source",
    "Depth",
    "Alpha",
    "Normal",
    "BaseColor",
    "Roughness",
    "Specular",
    "Metallic",
]

REQUIRED_PBR_PASSES = ["Normal", "BaseColor", "Roughness", "Specular", "Metallic"]

BACKDROP_GROUPS = {
    "Source": {
        "passes": ["Source"],
        "color": 0x5B7FA0FF,  # Blue
    },
    "Utility Passes": {
        "passes": ["Depth", "Alpha"],
        "color": 0x7F5B4BFF,  # Brown
    },
    "PBR Passes": {
        "passes": ["Normal", "BaseColor", "Roughness", "Specular", "Metallic"],
        "color": 0x4B7F52FF,  # Green
    },
}


def load_pbr_passes():
    """
    Opens a file/folder picker to select a directory or EXR frame file.
    Creates Read nodes for PNG sequences, MP4 files, or multi-layer EXR sequences.
    """
    # Open file picker dialog (allow both files and folders)
    selected_path = nuke.getFilename(
        "Select PBR Folder or EXR Frame", "*.exr *.png *.mp4", ""
    )

    if not selected_path:
        return  # User cancelled

    # Normalize path and detect type
    selected_path = selected_path.replace("\\", "/")

    if os.path.isfile(selected_path):
        # User selected a file - extract the folder
        folder_path = os.path.dirname(selected_path)
    elif os.path.isdir(selected_path):
        # User selected a folder - use directly
        folder_path = selected_path
    else:
        nuke.message("Error: Invalid path selected.")
        return

    # First, check for multi-layer EXR sequence (takes priority)
    exr_node, exr_type = _check_for_exr_sequence(folder_path)
    if exr_node:
        if exr_type == "multilayer":
            # Multi-layer EXR sequence found - create PBRController connected to it
            package_dir = os.path.dirname(os.path.dirname(__file__))
            control_gizmo_path = os.path.join(
                package_dir, "gizmos", "PBRController.gizmo"
            )
            control_gizmo_path = control_gizmo_path.replace("\\", "/")

            # Position PBRController below the EXR read node
            control_x = exr_node.xpos()
            control_y = exr_node.ypos() + 150

            pbr_controller = nuke.createNode(control_gizmo_path)
            pbr_controller.setXYpos(control_x, control_y)

            # Connect PBRController input to the EXR read node
            pbr_controller.setInput(0, exr_node)

            return
        # Single layer EXR will be handled below

    # Check if this is MP4 structure or subfolder structure
    has_mp4 = _check_for_mp4_passes(folder_path, PBR_PASSES)
    has_subfolders = any(
        os.path.isdir(os.path.join(folder_path, p)) for p in PBR_PASSES
    )

    if has_mp4:
        # Load MP4 files
        created_nodes = _load_mp4_passes(folder_path, PBR_PASSES)
    elif has_subfolders:
        # Check if required PBR passes exist before proceeding
        missing_required = []
        for pass_name in REQUIRED_PBR_PASSES:
            subfolder_path = os.path.join(folder_path, pass_name)
            if not os.path.exists(subfolder_path) or not os.path.isdir(subfolder_path):
                missing_required.append(pass_name)

        if missing_required:
            nuke.message(
                f"Error: Required PBR passes are missing:\n{', '.join(missing_required)}\n\nAll PBR passes (Normal, BaseColor, Roughness, Specular, Metallic) must be present."
            )
            return

        # Load PNG/EXR sequences from subfolders
        created_nodes = _load_subfolder_passes(folder_path, PBR_PASSES)
    else:
        nuke.message(
            "Error: No valid PBR pass structure found.\n\nSupported formats:\n1. EXR: folder with multi-layer EXR sequence (Frame_######.exr) \n\n2. PNG: folder with subfolders named by pass (PassName/...) \n\n3. MP4: folder with vidoes named by pass (PassName.mp4)"
        )
        return

    # Create backdrops for each group
    for backdrop_name, backdrop_info in BACKDROP_GROUPS.items():
        _create_backdrop(backdrop_name, backdrop_info, created_nodes)

    # Pack all passes using PBRPacker gizmo
    _create_pbr_packer(created_nodes, created_nodes.get("_max_x", 0), 0)


def _create_read_node_for_pass(folder_path, pass_name, x_pos, y_pos):
    """
    Creates a Read node for a specific PBR pass.

    Args:
        folder_path: Base folder containing pass subfolders
        pass_name: Name of the pass (e.g., "Normal", "BaseColor")
        x_pos: X position for the node
        y_pos: Y position for the node

    Returns:
        The created Read node or None if the pass doesn't exist
    """
    subfolder_path = os.path.join(folder_path, pass_name)

    # Check if subfolder exists
    if not os.path.exists(subfolder_path) or not os.path.isdir(subfolder_path):
        return None

    try:
        # Determine file extension based on pass type
        file_extension = ".exr" if pass_name == "Depth" else ".png"

        # Find sequence files
        files = os.listdir(subfolder_path)
        sequence_files = [f for f in files if f.lower().endswith(file_extension)]

        if not sequence_files:
            return None

        sequence_files.sort()
        first_file = sequence_files[0]

        # Detect frame numbering pattern and create sequence path
        full_path = os.path.join(subfolder_path, first_file)
        frame_numbers = []

        match = re.search(r"(\d+)(" + re.escape(file_extension) + r")$", first_file)
        if match:
            padding = len(match.group(1))
            sequence_pattern = re.sub(
                r"\d+" + re.escape(file_extension) + r"$",
                f"%0{padding}d{file_extension}",
                first_file,
            )
            full_path = os.path.join(subfolder_path, sequence_pattern).replace(
                "\\", "/"
            )

            # Extract frame numbers from all files
            for f in sequence_files:
                frame_match = re.search(r"(\d+)" + re.escape(file_extension) + r"$", f)
                if frame_match:
                    frame_numbers.append(int(frame_match.group(1)))

        # Create Read node
        read_node = nuke.createNode("Read")
        read_node["file"].setValue(full_path)

        # Set unique name to avoid conflicts when loading multiple times
        base_name = f"Read_{pass_name}"
        unique_name = base_name
        counter = 1
        while nuke.exists(unique_name):
            unique_name = f"{base_name}_{counter}"
            counter += 1

        read_node["name"].setValue(unique_name)
        read_node["label"].setValue(pass_name)

        # Set colorspace based on pass type
        if pass_name in ["Source", "BaseColor"]:
            read_node["colorspace"].setValue("sRGB")
        else:
            read_node["colorspace"].setValue("linear")

        # Set frame range if detected
        if frame_numbers:
            first_frame = min(frame_numbers)
            last_frame = max(frame_numbers)
            read_node["first"].setValue(first_frame)
            read_node["last"].setValue(last_frame)
            read_node["origfirst"].setValue(first_frame)
            read_node["origlast"].setValue(last_frame)

        # Set node position
        read_node.setXYpos(x_pos, y_pos)

        return read_node

    except Exception as e:
        nuke.message(f"Error loading {pass_name}: {str(e)}")
        return None


def _check_for_exr_sequence(folder_path):
    """
    Check if the folder contains a multi-layer EXR sequence.

    Args:
        folder_path: Path to the folder to check

    Returns:
        Tuple of (Read node or None, "multilayer" or "single" or None)
    """
    try:
        files = os.listdir(folder_path)
        exr_files = [f for f in files if f.lower().endswith(".exr")]

        if not exr_files:
            return None, None

        exr_files.sort()
        first_file = exr_files[0]

        # Check if it's a sequence pattern (frame_####.exr)
        match = re.search(r"(\d+)(\.exr)$", first_file, re.IGNORECASE)
        if not match:
            return None, None

        # Build sequence path
        padding = len(match.group(1))
        sequence_pattern = re.sub(
            r"\d+\.exr$", f"%0{padding}d.exr", first_file, flags=re.IGNORECASE
        )
        full_path = os.path.join(folder_path, sequence_pattern).replace("\\", "/")

        # Extract frame numbers
        frame_numbers = []
        for f in exr_files:
            frame_match = re.search(r"(\d+)\.exr$", f, re.IGNORECASE)
            if frame_match:
                frame_numbers.append(int(frame_match.group(1)))

        # Create Read node
        read_node = nuke.createNode("Read")
        read_node["file"].setValue(full_path)

        # Set unique name
        base_name = "Read_EXR_Sequence"
        unique_name = base_name
        counter = 1
        while nuke.exists(unique_name):
            unique_name = f"{base_name}_{counter}"
            counter += 1

        read_node["name"].setValue(unique_name)
        read_node["label"].setValue("Multi-Layer EXR")
        read_node["colorspace"].setValue("linear")

        # Set frame range
        if frame_numbers:
            first_frame = min(frame_numbers)
            last_frame = max(frame_numbers)
            read_node["first"].setValue(first_frame)
            read_node["last"].setValue(last_frame)
            read_node["origfirst"].setValue(first_frame)
            read_node["origlast"].setValue(last_frame)

        # Position node
        all_nodes = nuke.allNodes()
        if all_nodes:
            max_x = max(node.xpos() for node in all_nodes)
            read_node.setXYpos(max_x + 200, 0)
        else:
            read_node.setXYpos(0, 0)

        # Check if it's multi-layer by examining channels
        # Assume multi-layer if it's in the root folder (not in subfolders)
        return read_node, "multilayer"

    except Exception:
        return None, None


def _check_for_mp4_passes(folder_path, pbr_passes):
    """
    Check if the folder contains MP4 files for PBR passes.

    Args:
        folder_path: Path to the folder to check
        pbr_passes: List of expected pass names

    Returns:
        True if MP4 files are found, False otherwise
    """
    try:
        files = os.listdir(folder_path)
        mp4_files = [f for f in files if f.lower().endswith(".mp4")]

        # Check if any MP4 file matches a pass name
        for mp4_file in mp4_files:
            name_without_ext = os.path.splitext(mp4_file)[0]
            if name_without_ext in pbr_passes:
                return True

        return False
    except Exception:
        return False


def _load_mp4_passes(folder_path, pbr_passes):
    """
    Load MP4 files as PBR passes.

    Args:
        folder_path: Path to the folder containing MP4 files
        pbr_passes: List of expected pass names

    Returns:
        Dictionary of created Read nodes
    """
    created_nodes = {}

    # Calculate starting position
    all_nodes = nuke.allNodes()
    if all_nodes:
        max_x = max(node.xpos() for node in all_nodes)
        x_offset = max_x + 200
        y_offset = 0
    else:
        x_offset = 0
        y_offset = 0

    spacing = 125

    try:
        files = os.listdir(folder_path)

        for pass_name in pbr_passes:
            mp4_file = f"{pass_name}.mp4"

            if mp4_file in files or mp4_file.lower() in [f.lower() for f in files]:
                # Find the actual file (case-insensitive)
                actual_file = next(
                    (f for f in files if f.lower() == mp4_file.lower()), None
                )
                if not actual_file:
                    continue

                full_path = os.path.join(folder_path, actual_file).replace("\\", "/")

                # Create Read node
                read_node = nuke.createNode("Read")
                read_node["file"].setValue(full_path)

                # Set unique name
                base_name = f"Read_{pass_name}"
                unique_name = base_name
                counter = 1
                while nuke.exists(unique_name):
                    unique_name = f"{base_name}_{counter}"
                    counter += 1

                read_node["name"].setValue(unique_name)
                read_node["label"].setValue(pass_name)

                # Set colorspace
                if pass_name in ["Source", "BaseColor"]:
                    read_node["colorspace"].setValue("sRGB")
                else:
                    read_node["colorspace"].setValue("linear")

                # Set frame range for MP4 files
                read_node["first"].setValue(1)
                read_node["last"].setValue(100)
                read_node["origfirst"].setValue(1)
                read_node["origlast"].setValue(100)

                # Position node
                read_node.setXYpos(x_offset, y_offset)

                created_nodes[pass_name] = read_node
                x_offset += spacing

        created_nodes["_max_x"] = x_offset

    except Exception as e:
        nuke.message(f"Error loading MP4 passes: {str(e)}")

    return created_nodes


def _load_subfolder_passes(folder_path, pbr_passes):
    """
    Load PNG/EXR sequences from subfolders.

    Args:
        folder_path: Path to the folder containing pass subfolders
        pbr_passes: List of expected pass names

    Returns:
        Dictionary of created Read nodes
    """
    created_nodes = {}

    # Calculate starting position
    all_nodes = nuke.allNodes()
    if all_nodes:
        max_x = max(node.xpos() for node in all_nodes)
        x_offset = max_x + 200
        y_offset = 0
    else:
        x_offset = 0
        y_offset = 0

    spacing = 125

    for pass_name in pbr_passes:
        node = _create_read_node_for_pass(folder_path, pass_name, x_offset, y_offset)

        if node:
            created_nodes[pass_name] = node
            x_offset += spacing

    created_nodes["_max_x"] = x_offset

    return created_nodes


def _create_backdrop(name, backdrop_info, created_nodes):
    """
    Creates a backdrop node for a group of passes.

    Args:
        name: Name of the backdrop
        backdrop_info: Dictionary containing 'passes' list and 'color' value
        created_nodes: Dictionary of all created Read nodes
    """
    passes = backdrop_info["passes"]
    nodes = [
        created_nodes[pass_name] for pass_name in passes if pass_name in created_nodes
    ]

    if not nodes:
        return

    backdrop = nuke.nodes.BackdropNode()
    backdrop["label"].setValue(name)
    backdrop["tile_color"].setValue(backdrop_info["color"])
    backdrop["note_font_size"].setValue(20)

    # Calculate backdrop bounds
    min_x = min(node.xpos() for node in nodes)
    max_x = max(node.xpos() + node.screenWidth() for node in nodes)
    min_y = min(node.ypos() for node in nodes)
    max_y = max(node.ypos() + node.screenHeight() for node in nodes)

    backdrop["xpos"].setValue(min_x - 20)
    backdrop["ypos"].setValue(min_y - 80)
    backdrop["bdwidth"].setValue(max_x - min_x + 40)
    backdrop["bdheight"].setValue(max_y - min_y + 100)


def _create_pbr_packer(created_nodes, start_x, start_y):
    """
    Creates a PBRPacker gizmo to pack all PBR passes into one stream,
    then creates a PBRControl gizmo connected to it.

    Args:
        created_nodes: Dictionary of all created Read nodes
        start_x: X position to place the packer
        start_y: Y position to place the packer

    Returns:
        Tuple of (PBRPacker gizmo node, PBRControl gizmo node)
    """
    # Required inputs for PBRPacker: Normal, BaseColor, Roughness, Specular, Metallic
    required_passes = ["Normal", "BaseColor", "Roughness", "Specular", "Metallic"]

    # Check if all required passes exist
    missing_passes = [p for p in required_passes if p not in created_nodes]
    if missing_passes:
        nuke.message(
            f"Warning: Missing passes for PBRPacker: {', '.join(missing_passes)}"
        )
        return None, None

    # Create PBRPacker gizmo
    packer_y = start_y + 150
    packer_x = start_x - (len(required_passes) - 1) * 125 / 2  # Center under the nodes

    # Get the path to the gizmos (go up one level from beeble package)
    package_dir = os.path.dirname(os.path.dirname(__file__))
    packer_gizmo_path = os.path.join(package_dir, "gizmos", "PBRPacker.gizmo")
    packer_gizmo_path = packer_gizmo_path.replace("\\", "/")

    # Load and create the PBRPacker gizmo
    pbr_packer = nuke.createNode(packer_gizmo_path)
    pbr_packer.setXYpos(int(packer_x), int(packer_y))

    # Connect inputs: Normal, BaseColor, Roughness, Specular, Metallic (in order)
    pbr_packer.setInput(5, created_nodes["Source"])
    pbr_packer.setInput(4, created_nodes["Normal"])
    pbr_packer.setInput(3, created_nodes["BaseColor"])
    pbr_packer.setInput(2, created_nodes["Roughness"])
    pbr_packer.setInput(1, created_nodes["Specular"])
    pbr_packer.setInput(0, created_nodes["Metallic"])

    # Create PBRController gizmo below PBRPacker
    control_gizmo_path = os.path.join(package_dir, "gizmos", "PBRController.gizmo")
    control_gizmo_path = control_gizmo_path.replace("\\", "/")

    control_y = packer_y + 150
    pbr_controller = nuke.createNode(control_gizmo_path)
    pbr_controller.setXYpos(int(packer_x), int(control_y))

    # Connect PBRController input to PBRPacker output
    pbr_controller.setInput(0, pbr_packer)

    return pbr_packer, pbr_controller
