# beeble/lighting.py
"""Functions for directional and point lighting."""

import nuke
import os

from .utils import kelvin_to_rgb


def _update_light_color():
    """Update light color and UI visibility based on temperature mode."""
    node = nuke.thisNode()
    use_temp = node["use_temperature"].value()

    temp_knob = node["light_temperature"]
    color_knob = node["lightColor"]

    if use_temp:
        # Temperature mode: show slider, hide color picker
        temp_knob.setVisible(True)
        color_knob.setVisible(False)

        # Convert temperature to RGB and update light color
        temp = temp_knob.value()
        r, g, b = kelvin_to_rgb(temp)
        color_knob.setValue([r, g, b])
    else:
        # Color picker mode: hide slider, show color picker
        temp_knob.setVisible(False)
        color_knob.setVisible(True)


def _update_point_light_color():
    """Update point light color and UI visibility based on temperature mode."""
    node = nuke.thisNode()
    use_temp = node["use_temperature"].value()

    temp_knob = node["light_temperature"]
    color_knob = node["lightColor"]

    if use_temp:
        # Temperature mode: show slider, hide color picker
        temp_knob.setVisible(True)
        color_knob.setVisible(False)

        # Convert temperature to RGB and update light color
        temp = temp_knob.value()
        r, g, b = kelvin_to_rgb(temp)
        color_knob.setValue([r, g, b])
    else:
        # Color picker mode: hide slider, show color picker
        temp_knob.setVisible(False)
        color_knob.setVisible(True)


def _update_point_light_depth_from_rotation():
    """Update light depth based on rotation handle.

    Sensitivity: 180 degrees = 1 unit of depth change.
    Clockwise (negative rotation) = decrease depth (towards negative values).
    """
    node = nuke.thisNode()
    rotation = node["rotate"].value()

    # Convert rotation to depth: rotation / 180
    # Clockwise is negative in Nuke, so clockwise decreases depth
    depth = rotation / 180.0

    # Clamp to valid range
    depth = max(-5.0, min(5.0, depth))

    node["light_depth"].setValue(depth)


def add_directional_light():
    """
    Creates a DirectionalLight gizmo node in the node graph.

    Returns:
        The DirectionalLight gizmo node
    """
    # Get the path to the DirectionalLight gizmo (go up one level from beeble package)
    package_dir = os.path.dirname(os.path.dirname(__file__))
    gizmo_path = os.path.join(package_dir, "gizmos", "BeebleDirectionalLight.gizmo")
    gizmo_path = gizmo_path.replace("\\", "/")

    # Load and create the gizmo
    # All controls and knob ranges are embedded directly in the gizmo file
    directional_light = nuke.createNode(gizmo_path)

    return directional_light


def add_point_light():
    """
    Creates a PointLight gizmo node in the node graph.

    The point light position can be controlled by:
    - XY position: drag the translate handle in viewer, or edit in panel
    - Z depth: rotate the handle (clockwise = decrease depth, like a screwdriver),
      or edit directly in panel
      - 180 degrees of rotation = 1 unit of depth change
      - Depth range: -5 to 5

    Returns:
        The PointLight gizmo node
    """
    # Get the path to the PointLight gizmo (go up one level from beeble package)
    package_dir = os.path.dirname(os.path.dirname(__file__))
    gizmo_path = os.path.join(package_dir, "gizmos", "BeeblePointLight.gizmo")
    gizmo_path = gizmo_path.replace("\\", "/")

    # Load and create the gizmo
    # All controls and knob ranges are embedded directly in the gizmo file
    point_light = nuke.createNode(gizmo_path)

    # Initialize Transform handle translate to (0,0) which maps to image center
    # The expression offsets translate by half the image dimensions
    # so translate (0,0) = light at center (0.5, 0.5)
    point_light["translate"].setValue([0, 0])

    # Initialize rotation to 0
    point_light["rotate"].setValue(0)

    return point_light


def add_environment_light():
    """
    Creates an EnvironmentLight gizmo node in the node graph.

    The EnvironmentLight combines HDRI processing and IBL rendering into a single node.
    It wraps HDRIProcessor and IBLLight gizmos internally.

    All BlinkScript parameters and knob linkages are embedded directly in the gizmo files,
    so they persist across save/reload cycles.

    Inputs:
    - Input 0: PBR packed passes (Basecolor, Normal, Roughness, Metallic, Specular)
    - Input 1: Raw equirectangular HDRI

    Returns:
        The EnvironmentLight gizmo node
    """
    # Get the path to the EnvironmentLight gizmo (go up one level from beeble package)
    package_dir = os.path.dirname(os.path.dirname(__file__))

    gizmo_path = os.path.join(package_dir, "gizmos", "BeebleEnvironmentLight.gizmo")
    gizmo_path = gizmo_path.replace("\\", "/")

    # Load and create the gizmo
    # All parameters are embedded in the gizmo files (HDRIProcessor, IBLLight)
    # and knobs are linked via addUserKnob {41 ... T ...} syntax
    environment_light = nuke.createNode(gizmo_path)

    return environment_light
