import os
import bpy
from bpy.types import PropertyGroup
from bpy.props import (
    StringProperty,
    CollectionProperty,
    PointerProperty,
    FloatProperty,
    FloatVectorProperty,
    BoolProperty,
    EnumProperty,
    IntProperty,
)
import mathutils
from pathlib import Path
from ..core.shot_utils import (
    update_motion_strength,
    update_aspect_ratio,
    update_fps,
    update_resolution,
    update_use_depth_toggle,
    update_normal_strength,
    update_subdivision_level,
)


class BeebleCameraTransform(PropertyGroup):
    """Property group for storing a single 4x4 transform matrix"""

    # Store matrix as a flat array of 16 floats
    matrix: FloatVectorProperty(
        name="Transform Matrix",
        size=16,
        default=(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1),
        subtype="MATRIX",
        description="4x4 transform matrix stored as a flat array",
    )

    def set_matrix(self, matrix):
        """Set matrix from a mathutils.Matrix"""
        if not isinstance(matrix, mathutils.Matrix):
            raise TypeError("Input must be a mathutils.Matrix")
        # Flatten matrix to array
        self.matrix = [x for row in matrix for x in row]

    def get_matrix(self):
        """Get matrix as mathutils.Matrix"""
        return mathutils.Matrix(
            [
                self.matrix[0],
                self.matrix[1],
                self.matrix[2],
                self.matrix[3],
            ]
        )


class BeebleShotAttributes(PropertyGroup):
    """Property group for storing shot data references"""

    # Animation Actions
    lens_anim: PointerProperty(type=bpy.types.Action, name="Lens Animation")
    camera_anim: PointerProperty(type=bpy.types.Action, name="Camera Animation")
    footage_anim: PointerProperty(type=bpy.types.Action, name="Footage Animation")
    focus_anim: PointerProperty(type=bpy.types.Action, name="Focus Animation")

    # Image Maps
    source_map: PointerProperty(type=bpy.types.Image, name="Source Map")
    basecolor_map: PointerProperty(type=bpy.types.Image, name="BASECOLOR Map")
    normal_map: PointerProperty(type=bpy.types.Image, name="Normal Map")
    roughness_map: PointerProperty(type=bpy.types.Image, name="Roughness Map")
    specular_map: PointerProperty(type=bpy.types.Image, name="Specular Map")
    alpha_map: PointerProperty(type=bpy.types.Image, name="Alpha Map")
    depth_map: PointerProperty(type=bpy.types.Image, name="Depth Map")

    # Objects
    lens: PointerProperty(type=bpy.types.Camera, name="Lens")
    camera_anchor: PointerProperty(type=bpy.types.Object, name="Camera Anchor")
    footage_anchor: PointerProperty(type=bpy.types.Object, name="Footage Anchor")
    camera: PointerProperty(type=bpy.types.Object, name="Footage")
    footage: PointerProperty(type=bpy.types.Object, name="Footage")
    focus: PointerProperty(type=bpy.types.Object, name="Focus")

    # Materials and Modifiers
    material: PointerProperty(type=bpy.types.Material, name="Material")
    # Store the name of the modifier instead of the modifier itself
    delete_geometry_modifier_name: StringProperty(
        name="Delete Geometry Modifier Name", default=""
    )
    displace_geometry_modifier_name: StringProperty(
        name="Displace Geometry Modifier Name", default=""
    )


class BeebleShotProperties(PropertyGroup):
    """Properties for managing virtual set shots"""

    uid: StringProperty(name="Shot UID", description="Unique identifier for the shot")
    type: StringProperty(name="Shot Type", description="Type of the shot")
    set_type: StringProperty(name="Set Type", description="Type of the set")
    duration: FloatProperty(
        name="Duration",
        description="Duration of the shot in seconds",
        default=0.0,
        min=0.0,
    )
    frame_count: IntProperty(
        name="Frame Count",
        description="Number of frames in the shot",
    )
    frame_rate: StringProperty(
        name="Frame Rate", description="Frame rate of the shot", default="30"
    )
    file_name: StringProperty(name="Output Name", description="Name of the output file")
    local_path: StringProperty(
        name="Local Files Path", description="Path to local shot files", default=""
    )
    modified_at: StringProperty(
        name="Modified At", description="Last modification timestamp"
    )
    show_current_shot: BoolProperty(name="Show Current Shot", default=True)
    """
    Custom properties for edit settings
    """
    show_footage: BoolProperty(name="Show Footage Settings", default=True)
    show_camera: BoolProperty(name="Show Camera Settings", default=True)
    collection: PointerProperty(type=bpy.types.Collection, name="Collection")
    camera_transforms: CollectionProperty(
        type=BeebleCameraTransform, name="Camera Transform sequence"
    )
    attributes: PointerProperty(type=BeebleShotAttributes, name="Shot Attributes")
    show_camera_motion: BoolProperty(name="Show Camera Motion", default=False)
    motion_strength: FloatProperty(
        name="Effect Strength",
        description="Strength of smoothing or noise effect",
        default=0.0,
        min=0.0,
        max=1.0,
        update=lambda self, context: update_motion_strength(self, context),
    )
    aspect_ratio: EnumProperty(
        name="Aspect Ratio",
        description="Resolution Y is fixed regardless of what aspect ratio is used",
        items=[
            ("21:9", "21:9", ""),
            ("16:9", "16:9", ""),
            ("4:3", "4:3", ""),
            ("1:1", "1:1", ""),
            ("2:3", "2:3", ""),
            ("9:16", "9:16", ""),
        ],
        update=lambda self, context: update_aspect_ratio(self, context),
    )
    fps: EnumProperty(
        name="FPS",
        description="Frames per second",
        items=[
            ("24", "24 fps", ""),
            ("25", "25 fps", ""),
            ("30", "30 fps", ""),
            ("50", "50 fps", ""),
            ("60", "60 fps", ""),
            ("120", "120 fps", ""),
            ("240", "240 fps", ""),
            ("custom", "Custom", "Custom frames per second"),
        ],
        update=lambda self, context: update_fps(self, context),
    )
    custom_fps: IntProperty(
        name="Custom FPS",
        description="Custom frames per second",
        default=30,
        update=lambda self, context: update_fps(self, context),
    )
    resolution_x: IntProperty(
        name="Resolution X",
        description="Horizontal resolution of the camera",
    )
    resolution_y: IntProperty(
        name="Resolution Y",
        description="Vertical resolution of the camera",
        update=lambda self, context: update_resolution(self, context),
        min=4,
        subtype="PIXEL",
    )
    original_resolution_y: IntProperty(
        name="Original Resolution Y",
        description="Original vertical resolution of the camera",
    )
    show_save_settings: BoolProperty(name="Show Save Settings", default=True)
    use_iphone_video: BoolProperty(
        name="Use iPhone Video", description="Use iPhone video format", default=False
    )
    use_depth: BoolProperty(
        name="Use Depth",
        description="Toggle whether to use depth in 2D to 3D conversion",
        default=False,
        update=lambda self, context: update_use_depth_toggle(self, context),
    )
    normal_strength: FloatProperty(
        name="Normal Strength",
        description="Strength of the normal map",
        default=1.0,
        min=0.0,
        max=2.0,
        update=lambda self, context: update_normal_strength(self, context),
    )
    subdivision_level: IntProperty(
        name="Subdivision Level",
        description="Level of subdivision for the PlaneSubdivision_IMAGE modifier",
        default=0,
        min=0,
        max=7,
        update=lambda self, context: update_subdivision_level(self, context),
    )


class BeebleShotPanelProperties(PropertyGroup):
    """Properties for managing the shot panel UI state"""

    show_imported_shots: BoolProperty(name="Show imported shots", default=False)
    selected_shot: StringProperty(
        name="Selected Shot",
        description="UID of the currently selected shot",
        default="",
    )
    panel_state: EnumProperty(
        items=[
            ("GRID", "Grid View", "Show shots in grid layout"),
            ("DETAILS", "Detail View", "Show detailed view of selected shot"),
            ("EDIT", "Edit View", "Show edit interface for selected shot"),
        ],
        name="Panel State",
        description="Current state of the shots panel",
        default="GRID",
    )
    frame_start: IntProperty(
        name="Frame Start",
        description="Start frame for shots",
        default=0,
    )
    frame_end: IntProperty(
        name="Frame End",
        description="End frame for shots",
        default=250,
    )


class BeebleAccountSettings(PropertyGroup):
    user_id: StringProperty(
        name="User ID",
        description="Unique user identifier",
        default="",
        options={"SKIP_SAVE", "HIDDEN"},  # Don't save user ID between sessions
    )
    user_email: StringProperty(
        name="User Email",
        description="User email address",
        default="",
        options={"SKIP_SAVE", "HIDDEN"},  # Don't save email between sessions
    )
    addon_version: StringProperty(
        name="Addon Version", description="Current addon version", default="0.0.0"
    )
    login_state: EnumProperty(
        items=[
            ("LOGGED_OUT", "Logged Out", "User is not logged in"),
            ("LOGGING_IN", "Logging In", "Login in progress"),
            ("LOGGED_IN", "Logged In", "User is logged in"),
            ("NO_EARLY_ACCESS", "No Early Access", "User does not have early access"),
        ],
        name="Login State",
        default="LOGGED_OUT",
    )
    status_message: StringProperty(
        name="Status Message",
        description="Current login status message",
        default="Welcome! Please log in to continue.",
    )
    update_message: StringProperty(
        name="Update Message",
        description="Message about updates or new features",
        default="",
    )
    show_update_message: BoolProperty(
        name="Show Update Message",
        description="Whether to show the update message",
        default=False,
    )
    block_addon: BoolProperty(
        name="Block Addon",
        description="Whether to block the addon from being used",
        default=False,
    )
    workspace_path: StringProperty(
        name="Workspace Path",
        description="Directory for storing Beeble material (sets, shots, etc.)",
        subtype="DIR_PATH",
        default=os.path.join(str(Path.home()), "Beeble_Workspace"),
    )
    # New properties for auto-update
    latest_addon_version: StringProperty(
        name="Latest Addon Version",
        description="Latest available addon version",
        default="0.0.0",
    )
    update_available: BoolProperty(
        name="Update Available",
        description="Whether an update is available",
        default=False,
    )
    update_download_progress: FloatProperty(
        name="Update Download Progress",
        description="Progress of addon update download (0.0-1.0)",
        default=0.0,
        min=0.0,
        max=1.0,
    )
    update_status: StringProperty(
        name="Update Status", description="Current status of update process", default=""
    )
    update_state: EnumProperty(
        items=[
            ("NONE", "None", "No update in progress"),
            ("DOWNLOADING", "Downloading", "Update download in progress"),
            ("READY_TO_INSTALL", "Ready to Install", "Update ready to install"),
            ("INSTALLING", "Installing", "Update installation in progress"),
            ("COMPLETED", "Completed", "Update completed"),
            ("FAILED", "Failed", "Update failed"),
            (
                "CACHE_CLEANUP_COMPLETED",
                "Cache Cleanup Completed",
                "Cache cleanup process has finished",
            ),
        ],
        name="Update State",
        default="NONE",
    )
    downloaded_addon_path: StringProperty(
        name="Downloaded Addon Path",
        description="Path to downloaded addon zip file",
        default="",
    )
    active_tab: EnumProperty(
        items=[
            ("SHOTS", "Edit Shots", "Edit your shots", "OPTIONS", 2),
            ("SETTING", "My Setting", "Manage your settings", "USER", 3),
        ],
        name="Active Tab",
        description="Currently active tab",
        default="SHOTS",
    )


# List of all classes in this module
classes = (
    BeebleCameraTransform,
    BeebleShotAttributes,
    BeebleShotProperties,
    BeebleShotPanelProperties,
    BeebleAccountSettings,
)


def register():
    """Register property classes and create properties"""
    for cls in classes:
        bpy.utils.register_class(cls)

    # Register scene properties
    bpy.types.Scene.account_settings = PointerProperty(type=BeebleAccountSettings)
    bpy.types.Scene.beeble_imported_shots = CollectionProperty(
        type=BeebleShotProperties
    )
    bpy.types.Scene.beeble_shot_panel = PointerProperty(type=BeebleShotPanelProperties)


def unregister():
    """Unregister property classes and remove properties"""

    del bpy.types.Scene.account_settings
    del bpy.types.Scene.beeble_imported_shots
    del bpy.types.Scene.beeble_shot_panel

    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)


if __name__ == "__main__":
    register()
