class ShotsPanelMixin:
    def draw_shots_panel(self, context):
        layout = self.layout
        scene = context.scene
        shot_panel_props = scene.beeble_shot_panel

        # Import Shot button (opens file explorer)
        button_row = layout.row(align=True)
        button_row.scale_y = 1.5
        button_row.operator(
            "beeble.choose_shot_directory",
            text="Import VFX Passes (.zip or folder)",
            icon="ADD",
        )
        layout.separator()

        if shot_panel_props.show_imported_shots:
            # Import Shots section
            box = layout.box()
            row = box.row()
            row.label(text="Imported VFX Passes")

            # Validate and cleanup shots before drawing
            shots_to_remove = []
            shot_ids = [shot_temp.uid for shot_temp in scene.beeble_imported_shots]
            for shot_idx, shot_id in enumerate(shot_ids):
                # Check if any object with this UID exists in scene
                shot_exists = any(
                    obj
                    for obj in scene.objects
                    if "beeble_shot_uid" in obj.keys()
                    and obj["beeble_shot_uid"] == shot_id
                )
                if not shot_exists:
                    shots_to_remove.append(shot_idx)

            # Directory controls now inside the imported shots box
            layout.separator()

            # Single column layout for imported shots
            current_page_items = scene.beeble_imported_shots
            current_imported_shots = [
                shot_temp.uid for shot_temp in scene.beeble_imported_shots
            ]

            if len(current_imported_shots) - len(shots_to_remove) == 0:
                box_inner = box.box()
                col = box_inner.column()
                col.label(text="No Shots Imported")

            else:
                for idx, item in enumerate(current_page_items):
                    if item.uid in shot_ids and idx not in shots_to_remove:
                        self.draw_shot_grid(box, item)

                for item in current_page_items:
                    if shot_panel_props.selected_shot == item.uid:
                        self.draw_shot_details(context)
                        return

    def draw_shot_details(self, context):
        layout = self.layout
        scene = context.scene

        # Check if any object with this UID exists in scene
        shot_item = next(
            (
                shot_item
                for shot_item in scene.beeble_imported_shots
                if shot_item.uid == scene.beeble_shot_panel.selected_shot
            ),
            None,
        )

        shot_exists = None
        for obj in scene.objects:
            if (
                "beeble_shot_uid" in obj.keys()
                and obj["beeble_shot_uid"] == scene.beeble_shot_panel.selected_shot
            ):
                shot_exists = obj
                break

        if not shot_item or not shot_exists:
            # Instead of trying to modify properties, just draw a message and a back button
            row = layout.row(align=True)
            row.operator("beeble.close_shot_details", text="Back", icon="BACK")
            layout.label(text="Shot not found or was removed.")
            return

        """
        EDITING SHOT
        """
        # Main Edit section
        edit_box = layout.box()
        icon = "IMAGE_RGB" if shot_item.type == "image" else "VIEW_CAMERA"

        shot_collection = None
        for collection in context.scene.collection.children:
            if (
                "beeble_shot_uid" in collection.keys()
                and collection["beeble_shot_uid"] == shot_item.uid
            ):
                shot_collection = collection
                break

        display_name = shot_collection.name

        col = edit_box.column()
        row = col.row(align=True)

        left_side = row.row()
        left_side.alignment = "LEFT"
        left_side.operator(
            "beeble.active_shot", text=display_name, icon=icon, emboss=False
        ).uid = shot_item.uid

        # Footage settings nested box
        footage_box = edit_box.box()
        row = footage_box.row()
        row.label(text="Footage")

        # Footage properties
        col = footage_box.column(align=True)

        footage = next(
            (
                obj
                for obj in shot_item.collection.objects
                if "BeebleFootage." in obj.name
            ),
            None,
        )

        # Add Keying toggle
        row = col.row(align=True)
        row.prop(footage.active_material, '["keying"]', text="Keying")

        # Add Use Depth toggle at the top of the footage section
        row = col.row(align=True)
        split = row.split(factor=0.5)
        split.prop(shot_item, "use_depth", text="Use Depth")
        split.prop(footage, "visible_shadow", text="Cast Shadow")

        # Add original 2D to 3D slider - show only when depth is enabled
        if shot_item.use_depth:
            col.prop(footage, '["2d_to_3d"]', text="2D(0) ↔ 3D(1)")
            col.prop(shot_item, "subdivision_level", text="Subdivision Level")

        col.prop(
            footage.active_material,
            '["alpha_erosion"]',
            text="Alpha Erosion",
        )
        col.prop(
            footage.active_material,
            '["render_to_original"]',
            text="Render(0) ↔ Original(1)",
        )
        col.prop(
            footage.active_material,
            '["basecolor_to_original"]',
            text="BaseColor(0) ↔ Original(1)",
        )
        col.prop(shot_item, "normal_strength", text="Normal Strength")

        # Camera settings nested box
        camera_box = edit_box.box()
        row = camera_box.row()
        row.label(text="Camera")

        # Camera properties
        col = camera_box.column(align=True)

        cam_anchor = next(
            (
                obj
                for obj in shot_item.collection.objects
                if "BeebleCamAnchor." in obj.name
            ),
            None,
        )

        # Focal length
        if not shot_item.use_iphone_video:
            focal_length_row = col.row(align=True)
            focal_length_row.prop(cam_anchor, '["focal_length"]', text="Focal Length")

        # Zoom
        zoom_row = col.row(align=True)
        zoom_row.prop(cam_anchor, '["zoom_factor"]', text="Zoom")

        # Aperture
        camera = next(
            (obj for obj in shot_item.collection.objects if "BeebleCam." in obj.name),
            None,
        )
        col.prop(camera.data.dof, "aperture_fstop", text="Aperture (F-Stop)")

        # Camera Stabilization
        col.prop(shot_item, "motion_strength", text="Stabilization")

        # Export settings nested box
        export_box = edit_box.box()
        row = export_box.row()
        row.label(text="Render")

        # Export properties
        col = export_box.column(align=True)

        # Render Resolution controls
        col.label(text="Resolution")
        resolution_row = col.row(align=True)
        resolution_split = resolution_row.split(factor=0.5)
        x_row = resolution_split.row(align=True)
        x_split = x_row.split(factor=0.2)
        x_split.label(text="X:")
        x_split.prop(context.scene.render, "resolution_x", text="")
        y_row = resolution_split.row(align=True)
        y_split = y_row.split(factor=0.2)
        y_split.label(text="Y:")
        y_split.prop(context.scene.render, "resolution_y", text="")

        col.separator()

        # Filepath
        row = col.row(align=True)
        split = row.split(factor=0.3, align=True)
        split.label(text="Saved to")
        filepath_row = split.row(align=True)
        filepath_row.prop(context.scene.render, "filepath", text="")

        col.separator()

        # Format selection
        row = col.row(align=True)
        split = row.split(factor=0.3, align=True)
        split.label(text="Format")
        format_row = split.row(align=True)
        format_row.prop(context.scene.render.image_settings, "file_format", text="")

        if shot_item.type == "video":
            col.separator()
            # Frame range
            col.label(text="Frame Range")
            frame_row = col.row(align=True)
            frame_split = frame_row.split(factor=0.9)
            frame_col = frame_split.column()
            frame_text_split = frame_col.split(factor=0.5, align=True)
            frame_text_split.prop(context.scene, "frame_start", text="Start")
            frame_text_split.prop(context.scene, "frame_end", text="End")
            btn_col = frame_split.column()
            btn_col.operator("beeble.clear_frame", text="", icon="FILE_REFRESH")

        # Render button directly in the output box
        row = edit_box.row(align=True)
        row.scale_y = 1.5
        row.operator(
            "beeble.shot_export_settings",
            text="Render",
            icon="RENDER_STILL" if shot_item.type == "image" else "RENDER_ANIMATION",
        )
