import os # Add os import for path operations and size calculation

class SettingsPanelMixin:
    """Mixin class for drawing settings panel UI"""

    def _get_cache_size(self, workspace_path):
        """Helper function to calculate the size of cache directories and settings.json and return a formatted string."""
        total_size = 0
        directories_to_check = ["Temp", "Shots"]
        files_to_check = ["settings.json"]

        for directory in directories_to_check:
            dir_path = os.path.join(workspace_path, directory)
            if os.path.exists(dir_path):
                for dirpath, dirnames, filenames in os.walk(dir_path):
                    for f in filenames:
                        fp = os.path.join(dirpath, f)
                        if not os.path.islink(fp):
                            total_size += os.path.getsize(fp)

        for file_name in files_to_check:
            file_path = os.path.join(workspace_path, file_name)
            if os.path.exists(file_path) and not os.path.islink(file_path):
                total_size += os.path.getsize(file_path)

        size_mb = total_size / (1024 * 1024)

        if size_mb >= 1000:
            size_gb = size_mb / 1024
            return f"{size_gb:.2f} GB"
        else:
            return f"{size_mb:.2f} MB"


    def draw_user_info(self, layout, settings):
        """Draw user information section"""
        box = layout.box()
        col = box.column(align=True)
        col.separator(factor=0.5)

        col.label(text=f"Email: {settings.user_email}")

        col.separator(factor=0.3)

        logout_row = col.row()
        logout_row.scale_y = 1.2
        logout_row.operator("login.logout", text="Logout", icon='LOOP_BACK')

        col.separator(factor=0.5)


    def draw_update_section(self, layout, settings):
        """Draw addon update section"""
        box = layout.box()
        col = box.column(align=True)

        # Section title
        col.label(text="Addon Updates")
        col.separator(factor=0.5)

        # Check for updates button
        row = col.row(align=True)
        row.scale_y = 1.2
        row.operator("beeble.check_for_updates", text="Check for Updates", icon='FILE_REFRESH')

        col.separator(factor=0.3)

        # Current version info
        info_row = col.row()
        if settings.addon_version:
            info_row.label(text=f"Current version: {settings.addon_version}")
        else:
            info_row.label(text="Current version: Unknown")

        # If update is in progress, show status and progress (excluding cache cleanup state)
        if settings.update_state not in {'NONE', 'CACHE_CLEANUP_COMPLETED'}:
            # Status message
            col.separator()
            status_lines = settings.update_status.split('\n')
            for line in status_lines:
                if line.strip():  # Only add non-empty lines
                    status_row = col.row()
                    status_row.label(text=line)

            # Progress bar (if downloading or installing)
            if settings.update_state in {'DOWNLOADING', 'INSTALLING'}:
                progress_row = col.row()
                progress_row.prop(settings, "update_download_progress", text="Progress", slider=True)

            # Action buttons based on state
            col.separator()
            action_row = col.row(align=True)

            if settings.update_state == 'COMPLETED':
                # Quit button with red color
                action_row.scale_y = 1.5
                quit_row = action_row.row()
                quit_row.operator("beeble.quit_blender", text="Quit Blender", icon='QUIT')

            elif settings.update_state == 'FAILED':
                # Try again button
                action_row.scale_y = 1.2
                action_row.operator("beeble.check_for_updates", text="Try Again", icon='FILE_REFRESH')

        # If update is available but not in progress
        elif settings.update_available:
            col.separator()
            col.label(text=f"New version available: {settings.latest_addon_version}")

            col.separator(factor=0.3)

            # Download button
            download_row = col.row()
            download_row.scale_y = 1.2
            download_row.operator("beeble.download_update", text="Download & Install Update", icon='IMPORT')

        col.separator(factor=0.5)

    def draw_cache_management_section(self, layout, settings):
        """Draw cache management section"""
        box = layout.box()
        col = box.column(align=True)

        col.label(text="Cache Management")
        col.separator(factor=0.5)

        button = col.row(align=True)
        button.scale_y = 1.2
        button.operator("login.empty_workspace", text="Clean up Cache", icon='TRASH')

        if settings.workspace_path:
            col.separator(factor=0.3)
            cache_size_str = self._get_cache_size(settings.workspace_path)
            col.label(text=f"Cache size: {cache_size_str}")

            col.label(text="Cache directory:")
            col.label(text=settings.workspace_path)

            # Show cleanup success message and quit button if cache cleanup is completed
            if settings.update_state == 'CACHE_CLEANUP_COMPLETED':
                col.separator()
                status_lines = settings.update_status.split('\n')
                for line in status_lines:
                    if line.strip():
                        status_row = col.row()
                        status_row.label(text=line)

                col.separator(factor=0.3)
                action_row = col.row(align=True)
                action_row.scale_y = 1.5
                quit_row = action_row.row()
                quit_row.operator("beeble.quit_blender", text="Quit Blender", icon='QUIT')

        col.separator(factor=0.5)

    def draw_setting_panel(self, context):
        """Main draw method for settings panel with improved overall layout"""
        layout = self.layout
        settings = context.scene.account_settings

        # Add some top padding
        layout.separator(factor=0.5)

        self.draw_user_info(layout, settings)

        layout.separator()

        self.draw_update_section(layout, settings)

        layout.separator()

        self.draw_cache_management_section(layout, settings)

        # Support section with improved styling
        layout.separator()

        support_box = layout.box()

        # Section header
        header_row = support_box.row(align=True)
        header_row.label(text="Help & Resource")

        col = support_box.column(align=True)
        col.separator(factor=0.5)

        # Tutorial button
        tutorial_row = col.row()
        tutorial_row.scale_y = 1.2
        tutorial_op = tutorial_row.operator("beeble.open_web_url", text="Watch Tutorial", icon='PLAY')
        tutorial_op.url = "https://www.youtube.com/clip/UgkxjTyUo3AHb6TZ-L34bFIVBdyEoo8t6_aT"
        tutorial_op.description = "Watch our tutorial video to get started."

        col.separator(factor=0.3)

        # Discord button
        discord_row = col.row()
        discord_row.scale_y = 1.2
        discord_op = discord_row.operator("beeble.open_web_url", text="Join Discord Community", icon='URL')
        discord_op.url = "https://discord.gg/5REz3mzWwj"
        discord_op.description = "Join our Discord community for support and updates."

        col.separator(factor=0.3)

        # Feedback button
        feedback_row = col.row()
        feedback_row.scale_y = 1.2
        feedback_op = feedback_row.operator("beeble.open_web_url", text="Send Feedback", icon='OUTLINER_DATA_LIGHT')
        feedback_op.url = "https://app.beeble.ai/send-feedback"
        feedback_op.description = "Send us your feedback or report a bug."

        col.separator(factor=0.5)

        # Add some bottom padding
        layout.separator(factor=0.5)