import bpy
from bpy.types import Panel

from ...core.constants import TAB_CONFIG, WEBSITE_URL
from .shots_panel import ShotsPanelMixin
from .grid_panel import GridPanelMixin
from .settings_panel import SettingsPanelMixin

class BEEBLE_PT_MainPanel(Panel, GridPanelMixin, ShotsPanelMixin, SettingsPanelMixin):
    """Main panel containing tabs"""
    bl_label = "Beeble"
    bl_idname = "BEEBLE_PT_MainPanel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'Beeble'

    def draw(self, context):
        # Refresh session if needed
        layout = self.layout
        settings = context.scene.account_settings

        # Check if update is available but not in progress (but not on settings tab to avoid duplication)
        if settings.update_available and settings.update_state == 'NONE' and settings.active_tab != 'SETTING':
            # Draw update message box
            box = layout.box()

            # Title row
            title_row = box.row()
            title_row = title_row.column(align=True)
            title_row.alignment = 'CENTER'
            if settings.block_addon:
                title_row.alert = True
            title_row.label(text=f"New version available: {settings.latest_addon_version}")
            title_row.label(text=f"Current version: {settings.addon_version}")

            # Message box
            if settings.update_message != "":
                message_box = box.box()
                col = message_box.column(align=True)
                col.scale_y = 1.2
                if settings.block_addon:
                    col.alert = True
                for line in settings.update_message.split('\n'):
                    if line.strip():
                        col.label(text=f"{line.strip()}")

            # Download button
            button_row = box.row(align=True)
            button_row.alignment = 'EXPAND'
            button_row.scale_y = 1.5
            button_row.operator("beeble.download_update", text="Download the latest version", icon='IMPORT')

        # If update is in progress, show status (but not on settings tab to avoid duplication)
        elif settings.update_state not in {'NONE', 'CACHE_CLEANUP_COMPLETED'} and settings.active_tab != 'SETTING':
            box = layout.box()

            # Title
            title_row = box.row()
            title_row.alignment = 'CENTER'
            icon = 'CHECKMARK' if settings.update_state == 'COMPLETED' else 'LOOP_FORWARDS'
            title_row.label(text="Addon Update", icon=icon)

            # Status message
            status_box = box.box()
            col = status_box.column(align=True)
            status_lines = settings.update_status.split('\n')
            for line in status_lines:
                if line.strip():
                    col.label(text=line)

            # Progress bar for download/install
            if settings.update_state in {'DOWNLOADING', 'INSTALLING'}:
                box.prop(settings, "update_download_progress", text="Progress", slider=True)

            # Quit button if completed
            if settings.update_state == 'COMPLETED':
                quit_row = box.row(align=True)
                quit_row.alignment = 'EXPAND'
                quit_row.scale_y = 1.5
                quit_row.alert = True
                quit_row.operator("beeble.quit_blender", text="Quit Blender", icon='QUIT')

        # If addon is blocked, don't show anything
        if settings.block_addon:
            return

        # Check blender version
        blender_version = bpy.app.version
        if blender_version < (4, 2, 0):
            layout.label(text="Sorry! \nUpdate to 4.2.0 or later to use Beeble.")
            beeble_op = layout.operator("beeble.open_web_url", text="Visit Beeble", icon='URL')
            beeble_op.url = WEBSITE_URL
            beeble_op.description = "Join our waitlist and we'll let you in!"

            blender_op = layout.operator("beeble.open_web_url", text="Update Blender", icon='URL')
            blender_op.url = "https://www.blender.org/download/releases/4-2/"
            blender_op.description = "Download Blender 4.2.0 or later"

            return

        # Show login button if not logged in
        if settings.login_state != 'LOGGED_IN':
            if settings.login_state == 'LOGGING_IN':
                layout.label(text=settings.status_message)
                layout.operator("login.cancel_login", icon='X')

            else:
                for line in settings.status_message.split("\n"):
                    layout.label(text=line)
                layout.operator("login.browser_login", icon='URL')
            return

        # Get current panel name and icon
        current_tab = TAB_CONFIG[settings.active_tab]

        # Header row
        header_row = layout.row()

        # Left side - Current tab label
        label_split = header_row.split(factor=0.7)
        label_split.label(text=current_tab['label'], icon=current_tab['icon'])

        # Right side - Tab buttons
        button_split = label_split.split()
        button_row = button_split.row(align=True)
        button_row.alignment = 'RIGHT'

        # Create tab buttons
        for tab_id, config in TAB_CONFIG.items():
            op = button_row.operator("beeble.switch_tab", text="",
                icon=config['icon'],
                depress=(settings.active_tab == tab_id))
            op.panel = tab_id
            op.description = config['description']

        layout.separator()

        # Draw active panel content using a mapping of panel drawing methods
        panel_methods = {
            'SHOTS': self.draw_shots_panel,
            'SETTING': self.draw_setting_panel
        }
        panel_methods[settings.active_tab](context)