import os
import bpy

class GridPanelMixin:
    def draw_grid_panel(self, context, items, pagination, items_per_row, panel_type='set', parent=None):
        layout = parent if parent else self.layout
        # Draw pagination controls
        if pagination.get_total_pages(panel_type) > 1:
            row = layout.row(align=True)
            row.alignment = 'CENTER'

            # Previous page button
            prev_row = row.row()
            prev_row.enabled = pagination.current_page > 0
            op = prev_row.operator("beeble.change_page", text="", icon='TRIA_LEFT')
            op.direction = 'PREV'
            op.type = panel_type

            # Page indicator
            row.label(text=f"{pagination.current_page + 1} / {pagination.get_total_pages(panel_type)}")

            # Next page button
            next_row = row.row()
            next_row.enabled = pagination.current_page < pagination.get_total_pages(panel_type) - 1
            op = next_row.operator("beeble.change_page", text="", icon='TRIA_RIGHT')
            op.direction = 'NEXT'
            op.type = panel_type

        # Calculate pagination
        start_idx = pagination.current_page * pagination.items_per_page
        end_idx = min(start_idx + pagination.items_per_page, len(items))
        current_page_items = items[start_idx:end_idx]

        # Calculate rows needed
        num_items = len(current_page_items)
        rows_needed = (num_items + items_per_row - 1) // items_per_row

        # Draw grid
        for row_idx in range(rows_needed):
            main_row = layout.row()
            main_row.alignment = 'EXPAND'

            # Create splits for the row
            for col_idx in range(items_per_row - 1):
                split = main_row.split(factor=1.0 / (items_per_row - col_idx))
                idx = row_idx * items_per_row + col_idx
                if idx < num_items:
                    self.draw_grid_item(split.row(), current_page_items[idx], type=panel_type)
                main_row = split

            # Last column
            idx = row_idx * items_per_row + (items_per_row - 1)
            if idx < num_items:
                self.draw_grid_item(main_row, current_page_items[idx], type=panel_type)

    def draw_grid_item(self, row, item, type='shot'):
        # Create main box
        box = row.box()
        box.scale_y = 1.0
        box.scale_x = 1.2

        # Create main column
        main_col = box.column(align=True)
        main_col.scale_y = 0.9

        # Thumbnail section with overlaid icons
        thumb_col = main_col.column()

        # Create a container for the thumbnail and overlay
        container = thumb_col.column()

        filename = item.name
        display_name = filename.replace('_', ' ').title()

        box = container.row()
        col = box.column(align=True)
        col.label(text=display_name)

        if hasattr(item, 'license') or hasattr(item, 'author') or hasattr(item, 'source_url'):
            op = box.operator(
                "beeble.show_item_info",
                text="",
                icon='INFO',
                emboss=False
            )
            op.license = getattr(item, 'license', '')
            op.creator = getattr(item, 'creator', '')
            op.source_url = getattr(item, 'source_url', '')

        # Check if the item has a preview image attribute
        if hasattr(item, 'preview_image') and item.preview_image:
            item.preview_image.preview_ensure()
            container.template_icon(icon_value=item.preview_image.preview.icon_id, scale=4)
        else:
            # Show loading text instead of empty icon
            loading_col = container.column()
            loading_col.scale_y = 4  # Match the height of the icon
            loading_row = loading_col.row()
            loading_row.alignment = 'CENTER'
            loading_row.label(text="Loading...")
            loading_row.active = False


        main_col.separator(factor=1.5)

        # Bottom row with status and buttons
        bottom_row = main_col.row(align=True)
        bottom_row.scale_y = 1.0

        if type == 'world':
            op = bottom_row.operator(
                "beeble.select_world",
                text="Import",
                icon='IMPORT'
            )
            op.uid = item.uid

        elif type in ['object', 'interior']:
            op = bottom_row.operator(
                "beeble.add_object",
                text="Add",
                icon='ADD'
            )
            op.uid = item.uid
            op.type = type

    def draw_shot_grid(self, box, item):
        # Use template_icon again but with a smaller scale for the overlay
        shot_collection = None
        for collection in bpy.context.scene.collection.children:
            if "beeble_shot_uid" in collection.keys() and collection["beeble_shot_uid"] == item.uid:
                shot_collection = collection
                break

        display_name = shot_collection.name
        icon = 'IMAGE_RGB' if item.type == 'image' else 'VIEW_CAMERA'

        col = box.column()
        row = col.row(align=True)
        row.prop(item, "show_current_shot", icon="TRIA_DOWN" if item.show_current_shot else "TRIA_RIGHT", icon_only=True, emboss=False)

        left_side = row.row()
        left_side.alignment = 'LEFT'
        left_side.operator("beeble.active_shot", text=display_name, icon=icon, emboss=False).uid = item.uid

        split = row.split()
        right_side = split.row(align=True)
        right_side.alignment = 'RIGHT'

        if not item.show_current_shot and item.uid == bpy.context.scene.beeble_shot_panel.selected_shot:
            right_side.label(text="", icon='KEYTYPE_BREAKDOWN_VEC')

        else:
            right_side.label(text="", icon='BLANK1')

        layer_collection = bpy.context.view_layer.layer_collection.children.get(item.collection.name)
        is_hidden = layer_collection.hide_viewport if layer_collection else False
        icon = 'HIDE_ON' if is_hidden else 'HIDE_OFF'

        right_side.operator("beeble.toggle_shot_visibility", text="", icon=icon, emboss=False).shot_uid = item.uid
        right_side_split = right_side.split(align=True)
        right_side_split.operator("beeble.remove_shot_from_scene", text="", icon='TRASH', emboss=False).shot_uid = item.uid

        if item.show_current_shot:
            # Second row just for duration if it's a video
            if item.type == "video":
                duration_row = col.row(align=True)
                duration_row.alignment = 'LEFT'
                duration_row.scale_y = 0.8
                duration_row.label(text=f"Duration: {item.duration:.2f}s ({item.frame_count} frames)", icon='LAYER_USED') #DOT
                duration_row.active = False

            # Use different icons based on visibility
            status_row = col.row(align=True)

            op = status_row.operator("beeble.edit_shot", text="Click to Edit & Render", icon='GREASEPENCIL', depress=(bpy.context.scene.beeble_shot_panel.selected_shot == item.uid))
            op.shot_uid = item.uid
            status_row.scale_y = 1.2