import bpy
from bpy.types import Operator

from ...core.api import get_api_client
from ...core.logger import logger

import webbrowser

class BEEBLE_OT_OpenWebUrl(Operator):
    bl_idname = "beeble.open_web_url"
    bl_label = "Open url"
    bl_description = "Open in default web browser"
    url: bpy.props.StringProperty(default="")
    description: bpy.props.StringProperty(default="")

    @classmethod
    def description(cls, context, properties):
        if properties.description:
            return properties.description
        return "Open in the default web browser"

    def execute(self, context):
        try:
            logger.info(f"Opening URL: {self.url}")
            webbrowser.open(self.url)
        except Exception as e:
            logger.error(f"Error opening URL: {str(e)}")

        return {'FINISHED'}


class BEEBLE_OT_SwitchTab(Operator):
    bl_idname = "beeble.switch_tab"
    bl_label = ""
    bl_options = {'INTERNAL'}

    panel: bpy.props.StringProperty()
    description: bpy.props.StringProperty(
        name="Description",
        description="Description of the tab's functionality",
        default=""
    )

    @classmethod
    def description(cls, context, properties):
        return properties.description

    def execute(self, context):
        context.scene.account_settings.active_tab = self.panel

        # Track event
        if self.panel == "IMPORT":
            get_api_client().track_event("blender_import_clicked")

        return {'FINISHED'}

class BEEBLE_OT_InfoAlert(bpy.types.Operator):
    """Alert for invalid file selection"""
    bl_idname = "beeble.info_alert"
    bl_label = "Info"

    message: bpy.props.StringProperty(
        name="Message",
        description="The message to display in the dialog",
        default=""
    )

    def draw_menu(self, menu, context):
        layout = menu.layout
        col = layout.column(align=True)
        col.scale_y = 1.2
        col.label(text=self.message, icon='INFO')

    def execute(self, context):
        logger.info(f"Info Alert: {self.message}")
        return {'FINISHED'}

    def invoke(self, context, event):
        context.window_manager.popup_menu(self.draw_menu, title="", icon='NONE')
        return {'FINISHED'}


class BEEBLE_OT_ErrorAlert(bpy.types.Operator):
    """Alert for invalid file selection"""
    bl_idname = "beeble.error_alert"
    bl_label = "Error"

    message: bpy.props.StringProperty(
        name="Error Message",
        description="The message to display in the error dialog",
        default="An error occurred."
    )

    def draw(self, context):
        layout = self.layout
        col = layout.column(align=True)
        col.scale_y = 1.2

        for line in self.message.split("\n"):
            col.label(text=line)

    def execute(self, context):
        logger.info(f"Error Alert: {self.message}")
        return {'FINISHED'}

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self,
            width=400,
            title="Error")