import os
import glob
import uuid
import math
import json
import shutil
import zipfile

import bpy
from bpy.types import Operator
from bpy.props import StringProperty

from ...core.api import get_api_client
from ...core.logger import logger
from ...core.helpers import purge_unused_data
from ...core.shot_utils import (
    ensure_template_exists,
    find_objects_from_template,
    append_collection,
    apply_autofocus,
    apply_foot_reference,
    apply_camera_tracking,
    apply_focal_length_tracking,
    apply_pbr_maps,
    apply_geometry,
    calculate_relative_camera_transform,
    apply_depth_min_max,
    apply_lens_tracking_legacy,
    apply_focal_length_tracking_legacy,
    apply_camera_tracking_legacy,
    apply_autofocus_legacy,
    apply_foot_reference_legacy,
    apply_depth_min_max_legacy,
)


class BEEBLE_OT_RemoveShotFromScene(Operator):
    bl_idname = "beeble.remove_shot_from_scene"
    bl_label = "Do you want to remove this shot from the scene?"
    bl_description = "Remove this shot from scene and cleanup all related data"
    shot_uid: bpy.props.StringProperty()

    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)

    def cleanup_related_data(self, shot_item):
        """Clean up all data blocks related to the shot"""
        shot_attributes = shot_item.attributes

        # Clean up animations
        animations = [
            shot_attributes.lens_anim,
            shot_attributes.camera_anim,
            shot_attributes.footage_anim,
            shot_attributes.focus_anim,
        ]
        for action in animations:
            if action:
                bpy.data.actions.remove(action)

        # Clean up all maps
        maps = [
            shot_attributes.depth_map,
            shot_attributes.source_map,
            shot_attributes.basecolor_map,
            shot_attributes.normal_map,
            shot_attributes.roughness_map,
            shot_attributes.specular_map,
            shot_attributes.alpha_map,
        ]
        for image in maps:
            if image:
                bpy.data.images.remove(image)

        # Clean up materials
        if shot_attributes.material:
            if isinstance(shot_attributes.material, bpy.types.Material):
                bpy.data.materials.remove(shot_attributes.material)
            # If it's an object containing materials
            elif isinstance(shot_attributes.material, bpy.types.Object):
                for mat_slot in shot_attributes.material.material_slots:
                    if mat_slot.material:
                        bpy.data.materials.remove(mat_slot.material)

        # Handle footage cleanup with complete geometry nodes and mesh removal
        if shot_attributes.footage and isinstance(
            shot_attributes.footage, bpy.types.Object
        ):
            # Clean up geometry modifiers and their node groups
            for modifier in shot_attributes.footage.modifiers:
                if modifier.type == "NODES":
                    # Remove the geometry nodes tree if it exists
                    if modifier.node_group:
                        node_tree = modifier.node_group
                        bpy.data.node_groups.remove(node_tree, do_unlink=True)
                shot_attributes.footage.modifiers.remove(modifier)

            # Remove specific named modifiers
            if shot_attributes.delete_geometry_modifier_name:
                modifier = shot_attributes.footage.modifiers.get(
                    shot_attributes.delete_geometry_modifier_name
                )
                if modifier:
                    if modifier.type == "NODES" and modifier.node_group:
                        bpy.data.node_groups.remove(modifier.node_group, do_unlink=True)
                    shot_attributes.footage.modifiers.remove(modifier)

            if shot_attributes.displace_geometry_modifier_name:
                modifier = shot_attributes.footage.modifiers.get(
                    shot_attributes.displace_geometry_modifier_name
                )
                if modifier:
                    if modifier.type == "NODES" and modifier.node_group:
                        bpy.data.node_groups.remove(modifier.node_group, do_unlink=True)
                    shot_attributes.footage.modifiers.remove(modifier)

            # Unlink footage from all collections
            for col in shot_attributes.footage.users_collection:
                col.objects.unlink(shot_attributes.footage)

            # Remove footage and its mesh data
            if shot_attributes.footage.data:
                mesh_data = shot_attributes.footage.data
                # Remove the object first
                bpy.data.objects.remove(shot_attributes.footage, do_unlink=True)
                # Then remove the mesh data
                if isinstance(mesh_data, bpy.types.Mesh):
                    if (
                        mesh_data.users == 0
                    ):  # Only remove if no other objects are using it
                        bpy.data.meshes.remove(mesh_data)

        # Clean up anchor objects
        for anchor in [shot_attributes.camera_anchor, shot_attributes.footage_anchor]:
            if anchor and isinstance(anchor, bpy.types.Object):
                if anchor.name in bpy.data.objects:
                    bpy.data.objects.remove(anchor, do_unlink=True)

        # Clean up camera and focus
        for obj, obj_type in [
            (shot_attributes.camera, "CAMERA"),
            (shot_attributes.focus, "EMPTY"),
        ]:
            if obj and isinstance(obj, bpy.types.Object):
                if obj_type == "CAMERA" and obj.data:
                    camera_data = obj.data
                    bpy.data.objects.remove(obj, do_unlink=True)
                    bpy.data.cameras.remove(camera_data)
                else:
                    if obj.name in bpy.data.objects:
                        bpy.data.objects.remove(obj, do_unlink=True)

        # Handle lens specifically
        if shot_attributes.lens:
            if isinstance(shot_attributes.lens, bpy.types.Camera):
                bpy.data.cameras.remove(shot_attributes.lens)
            elif isinstance(shot_attributes.lens, bpy.types.Object):
                if shot_attributes.lens.data and isinstance(
                    shot_attributes.lens.data, bpy.types.Camera
                ):
                    camera_data = shot_attributes.lens.data
                    bpy.data.objects.remove(shot_attributes.lens, do_unlink=True)
                    bpy.data.cameras.remove(camera_data)

    def execute(self, context):
        logger.info(f"Remove {self.shot_uid} from scene")
        shot_index, shot_item = next(
            (
                (idx, shot)
                for idx, shot in enumerate(context.scene.beeble_imported_shots)
                if shot.uid == self.shot_uid
            ),
            (-1, None),
        )

        if shot_item:
            shot_collection = shot_item.collection
            if shot_collection:
                # Remove all objects in collection recursively
                def remove_collection_objects(collection):
                    for obj in list(collection.objects):
                        # Clean up geometry nodes for each object
                        if obj.type == "MESH":
                            for modifier in obj.modifiers:
                                if modifier.type == "NODES":
                                    if modifier.node_group:
                                        node_tree = modifier.node_group
                                        bpy.data.node_groups.remove(
                                            node_tree, do_unlink=True
                                        )
                                obj.modifiers.remove(modifier)

                        # Unlink from collections
                        for col in obj.users_collection:
                            col.objects.unlink(obj)

                        # Remove object and its data
                        if obj.name in bpy.data.objects:
                            mesh_data = obj.data
                            bpy.data.objects.remove(obj, do_unlink=True)
                            if (
                                isinstance(mesh_data, bpy.types.Mesh)
                                and mesh_data.users == 0
                            ):
                                bpy.data.meshes.remove(mesh_data)

                    # Handle child collections
                    for child in list(collection.children):
                        remove_collection_objects(child)
                        collection.children.unlink(child)
                        bpy.data.collections.remove(child)

                # Remove objects and unlink collection
                remove_collection_objects(shot_collection)
                context.scene.collection.children.unlink(shot_collection)

                # Clean up all related data
                self.cleanup_related_data(shot_item)

                # Finally remove the main collection
                bpy.data.collections.remove(shot_collection)

                logger.info(
                    f"Successfully removed shot {self.shot_uid} and cleaned up related data"
                )

            # Remove from beeble_imported_shots
            context.scene.beeble_imported_shots.remove(shot_index)
            logger.info(f"Removed shot from beeble_imported_shots")
            purge_unused_data()

            return {"FINISHED"}

        else:
            logger.error(f"Failed to remove shot {self.shot_uid}")
            return {"CANCELLED"}


class BEEBLE_OT_ImportShot(Operator):
    """Base class for shot importing operations"""

    bl_idname = "beeble.import_shot"
    bl_label = "Import Beeble 3D Shot"
    bl_options = {"REGISTER", "UNDO"}

    shot_path: StringProperty(
        name="Shot Path", description="Path to the shot directory", default=""
    )

    def get_image_info(self, shot_path):
        """
        Get frame count and resolution from source images or basecolor using Blender's native image loading.
        First checks source path, then falls back to basecolor if no source found.

        Args:
            shot_path: Path to the shot directory

        Returns:
            tuple: (frame_count, width, height)

        Raises:
            RuntimeError: If neither source nor basecolor images are found
        """
        # First try source path
        # Check for numbered image sequence (like 000000.png/0000.png, 000001.png/0001.png, etc)
        sequence_files = sorted(glob.glob(os.path.join(shot_path, "source", "*.png")))
        if sequence_files and sequence_files[0]:
            # Get resolution from first frame using Blender's image load
            temp_img = bpy.data.images.load(sequence_files[0], check_existing=False)
            width, height = temp_img.size
            bpy.data.images.remove(temp_img)  # Clean up temporary image
            return len(sequence_files), width, height

        # Check for single source image
        single_file = os.path.join(shot_path, "source.png")
        if os.path.exists(single_file):
            temp_img = bpy.data.images.load(single_file, check_existing=False)
            width, height = temp_img.size
            bpy.data.images.remove(temp_img)  # Clean up temporary image
            return 1, width, height

        # If no source found, try basecolor path
        # Check for basecolor sequence
        basecolor_sequence = sorted(
            glob.glob(os.path.join(shot_path, "basecolor", "*.png"))
        )
        if basecolor_sequence and basecolor_sequence[0]:
            temp_img = bpy.data.images.load(basecolor_sequence[0], check_existing=False)
            width, height = temp_img.size
            bpy.data.images.remove(temp_img)
            return len(basecolor_sequence), width, height

        # Check for single basecolor image
        basecolor_file = os.path.join(shot_path, "basecolor.png")
        if os.path.exists(basecolor_file):
            temp_img = bpy.data.images.load(basecolor_file, check_existing=False)
            width, height = temp_img.size
            bpy.data.images.remove(temp_img)
            return 1, width, height

        raise RuntimeError("No source or basecolor images found in the shot path")

    def get_default_camera_data(self, shot_path):
        """
        Returns default camera data when camera.json is missing
        Extends frames based on number of source images if available
        Uses resolution from source images if available
        """
        frame_count, width, height = self.get_image_info(shot_path)

        # Determine lens and sensor height based on resolution
        if height > width:
            orientation = 1  # Portrait
            sensor_height = 36.0
            focal_length = 24.0
            transform = [[1, 0, 0, 0], [0, 1, 0, 0], [0, 0, 1, 1], [0, 0, 0, 1]]
            studio_transform = [[1, 0, 0, 0], [0, 0, 1, 0], [0, -1, 0, 0], [0, 0, 0, 1]]
        else:
            orientation = 3  # Landscape
            sensor_height = 24.0
            focal_length = 28.0
            transform = [[0, -1, 0, 0], [1, 0, 0, 0], [0, 0, 1, 1], [0, 0, 0, 1]]
            studio_transform = [[1, 0, 0, 0], [0, 0, 1, 0], [0, -1, 0, 0], [0, 0, 0, 1]]

        # Default single frame camera data
        default_frame = {
            "focal_length": focal_length,
            "transform": transform,
            "position_median": [0, 0, -1],  # Center focus point
        }

        # Create frames array with correct length
        frames = [default_frame.copy() for _ in range(frame_count)]

        return {
            "resolution": [width, height],
            "frame_count": frame_count,
            "orientation": orientation,
            "sensor_height": sensor_height,
            "scale": 1.0,
            "studio": {"transform": studio_transform},
            "frames": frames,
        }

    def set_camera_view(self, context, collection):
        # Find the camera in the collection
        shot_camera = None
        for obj in collection.objects:
            if obj.type == "CAMERA" and "BeebleCam." in obj.name:
                shot_camera = obj
                break
        if shot_camera:
            # Set the active camera
            context.scene.camera = shot_camera

            # Change viewport to camera view
            for area in context.screen.areas:
                if area.type == "VIEW_3D":
                    for space in area.spaces:
                        if space.type == "VIEW_3D":
                            space.region_3d.view_perspective = "CAMERA"
                            break

    def initiate(self, context):
        # Remove the object from the collection property
        shots_to_remove = []
        shot_ids = [shot_temp.uid for shot_temp in context.scene.beeble_imported_shots]
        for shot_id in shot_ids:
            # Check if any object with this UID exists in scene
            shot_exists = any(
                obj
                for obj in context.scene.objects
                if "beeble_shot_uid" in obj.keys() and obj["beeble_shot_uid"] == shot_id
            )
            if not shot_exists:
                shots_to_remove.append(shot_id)

        for shot_id in shots_to_remove:
            for idx, item in enumerate(context.scene.beeble_imported_shots):
                if item.uid == shot_id:
                    context.scene.beeble_imported_shots.remove(idx)
                    break

    def execute(self, context):
        try:
            self.initiate(context)
            logger.info("execute 1")
            shot_uid = str(uuid.uuid4())

            # Load and validate camera data
            camera_data_path = os.path.join(self.shot_path, "camera.json")
            using_default_camera = False

            if os.path.exists(camera_data_path):
                with open(camera_data_path, "r") as f:
                    camera_data = json.load(f)
            else:
                try:
                    camera_data = self.get_default_camera_data(self.shot_path)
                    using_default_camera = True
                    frame_count = camera_data["frame_count"]
                    resolution = camera_data["resolution"]
                    warning_msg = f"Camera data not found, using defaults ("
                    if frame_count > 1:
                        warning_msg += f"{frame_count} frames, "
                    warning_msg += f"{resolution[0]}x{resolution[1]})"
                    logger.warning(warning_msg)
                except Exception as e:
                    logger.error(f"Failed to set default camera data: {str(e)}")
                    camera_data = self.get_default_camera_data(
                        ""
                    )  # Empty path to force defaults
            logger.info("execute 2")

            # Determine shot type early to use in conflict detection
            shot_type = "image" if camera_data.get("frame_count") == 1 else "video"

            # Check for conflicts between current scene settings and imported data
            current_fps = context.scene.render.fps
            imported_fps = int(round(camera_data.get("frame_rate")))
            current_resolution_x = context.scene.render.resolution_x
            current_resolution_y = context.scene.render.resolution_y
            imported_resolution_x = camera_data.get("resolution")[0]
            imported_resolution_y = camera_data.get("resolution")[1]

            # Check if conflicts exist and no temporary user choices are stored
            # Skip FPS conflict for image shots since images don't have frame rates
            has_fps_conflict = shot_type != "image" and current_fps != imported_fps
            has_resolution_conflict = (
                current_resolution_x != imported_resolution_x
                or current_resolution_y != imported_resolution_y
            )

            # If there are conflicts and no user choices stored, show dialog
            if (has_fps_conflict or has_resolution_conflict) and (
                "beeble_temp_use_imported_fps" not in context.scene
                and "beeble_temp_use_imported_resolution" not in context.scene
            ):
                # Show conflict dialog
                bpy.ops.beeble.import_settings_conflict_dialog(
                    "INVOKE_DEFAULT",
                    current_fps=current_fps,
                    imported_fps=imported_fps,
                    current_resolution_x=current_resolution_x,
                    current_resolution_y=current_resolution_y,
                    imported_resolution_x=imported_resolution_x,
                    imported_resolution_y=imported_resolution_y,
                    shot_path=self.shot_path,
                    shot_type=shot_type,
                )
                return {"FINISHED"}  # Dialog will handle the import
            logger.info("execute 3")

            # Import template from shot directory
            workspace_path = bpy.path.abspath(
                context.scene.account_settings.workspace_path
            )
            template_path = ensure_template_exists(workspace_path)

            if not template_path:
                logger.error("Failed to find or download template file")
                return {"CANCELLED"}

            _, collection_name = append_collection(template_path, auto_remove=False)
            # Setup scene resolution from camera data
            # try:
            #     context.scene.render.resolution_x = camera_data.get('resolution')[0]
            #     context.scene.render.resolution_y = camera_data.get('resolution')[1]
            # except Exception as e:
            #     logger.error(f"Failed to set resolution: {str(e)}")
            #     return {'CANCELLED'}
            logger.info("execute 4")

            # Setup scene frame range
            try:
                frame_count = camera_data.get("frame_count")
                context.scene.frame_start = 0
                context.scene.frame_current = 0
            except Exception as e:
                logger.error(f"Failed to set frame range: {str(e)}")
                return {"CANCELLED"}

            # Get collection
            use_video_atlas = (
                True if "output.mp4" in os.listdir(self.shot_path) else False
            )
            if not os.path.exists(os.path.join(self.shot_path, "Depth.mp4")):
                if not os.path.exists(os.path.join(self.shot_path, "Depth/Depth_000001.exr")):
                    depth_dir = os.path.join(self.shot_path, "depth")
                    if not os.path.exists(depth_dir):
                        depth_dir = os.path.join(self.shot_path, "depth_norm")
                    # Check for both 6-digit and 4-digit formats for backward compatibility
                    depth_files = os.listdir(depth_dir)
                    use_position_map = not use_video_atlas and (
                        True if ("000000.exr" not in depth_files and "0000.exr" not in depth_files) else False
                    )
                    format_after_1_10_0 = False
                else:
                    use_position_map = False
                    format_after_1_10_0 = True
            else:
                use_position_map = False
                format_after_1_10_0 = True
            collection = find_objects_from_template(
                collection_name, use_video_atlas, use_position_map
            )
            collection_suffix = collection_name.split(".")[-1]

            collection["camera_anchor"]["resolution_W"] = camera_data.get("resolution")[
                0
            ]
            collection["camera_anchor"]["resolution_H"] = camera_data.get("resolution")[
                1
            ]
            collection["camera_anchor"]["focal_length"] = camera_data.get("frames")[0][
                "focal_length"
            ]
            collection["camera_anchor"]["sensor_size"] = camera_data.get(
                "sensor_height"
            )
            collection["footage"]["depth_frame_offset"] = 1 if format_after_1_10_0 else 0

            # set collection properties
            bpy.data.collections.get(collection_name)["frame_start"] = 0
            bpy.data.collections.get(collection_name)["frame_end"] = (
                len(camera_data.get("frames")) - 1
            )
            context.scene.frame_end = bpy.data.collections.get(collection_name)[
                "frame_end"
            ]

            use_iphone_video = (
                True if camera_data.get("frames")[0].get("camera_given") else False
            )

            # Set transform and rotation for BeebleAnchor
            if not use_iphone_video:
                collection["global_anchor"].location[2] = 1.7
                collection["global_anchor"].rotation_euler[0] = math.radians(90)
            logger.info("execute 5")

            # Apply all modifications
            if bpy.app.version >= (5, 0, 0):
                if not all([
                    apply_focal_length_tracking(context.scene, collection["camera_anchor"], camera_data, use_iphone_video),
                    apply_camera_tracking(context.scene, collection["camera_anchor"], camera_data, use_iphone_video),
                    apply_camera_tracking(context.scene, collection["footage_anchor"], camera_data, use_iphone_video),
                    apply_pbr_maps(collection["material"], shot_type, self.shot_path, camera_data, collection_suffix, use_video_atlas),
                    apply_geometry(collection["geometry_modifiers"], collection["footage"], shot_type, self.shot_path, camera_data, collection_suffix),
                    apply_autofocus(context.scene, camera_data, collection["focus"], use_iphone_video),
                    apply_foot_reference(context.scene, camera_data, collection["foot_reference"], use_iphone_video),
                    apply_depth_min_max(context.scene, camera_data, collection["footage"])
                ]):
                    logger.error("Failed to apply all modifications")
                    return {"CANCELLED"}
            else:
                if not all([
                    apply_focal_length_tracking_legacy(context.scene, collection["camera_anchor"], camera_data, use_iphone_video),
                    apply_camera_tracking_legacy(context.scene, collection["camera_anchor"], camera_data, use_iphone_video),
                    apply_camera_tracking_legacy(context.scene, collection["footage_anchor"], camera_data, use_iphone_video),
                    apply_pbr_maps(collection["material"], shot_type, self.shot_path, camera_data, collection_suffix, use_video_atlas),
                    apply_geometry(collection["geometry_modifiers"], collection["footage"], shot_type, self.shot_path, camera_data, collection_suffix),
                    apply_autofocus_legacy(context.scene, camera_data, collection["focus"], use_iphone_video),
                    apply_foot_reference_legacy(context.scene, camera_data, collection["foot_reference"], use_iphone_video),
                    apply_depth_min_max_legacy(context.scene, camera_data, collection["footage"])
                ]):
                    logger.error("Failed to apply all modifications")
                    return {"CANCELLED"}

            # Add to imported shots
            new_shot = context.scene.beeble_imported_shots.add()

            # Store additional metadata
            new_shot.uid = shot_uid
            new_shot.type = shot_type
            new_shot.set_type = (
                camera_data.get("studio")["type"]
                if "type" in camera_data.get("studio")
                else "unknown"
            )
            new_shot.local_path = self.shot_path
            new_shot.collection = bpy.data.collections.get(collection_name)
            duration = camera_data.get("frame_count") / camera_data.get("frame_rate")
            new_shot.duration = round(duration, 2)

            for frame_data in camera_data.get("frames"):
                new_transform = new_shot.camera_transforms.add()
                # Store relative camera transform instead of raw transform
                relative_transform = calculate_relative_camera_transform(
                    frame_data["transform"], camera_data
                )
                new_transform.set_matrix(relative_transform)

            new_shot.frame_count = camera_data.get("frame_count")
            new_shot.frame_rate = str(round(camera_data.get("frame_rate")))
            new_shot.resolution_x = camera_data.get("resolution")[0]
            new_shot.resolution_y = camera_data.get("resolution")[1]
            new_shot.original_resolution_y = camera_data.get("resolution")[1]

            new_shot.file_name = os.path.basename(self.shot_path)
            new_shot.use_iphone_video = use_iphone_video

            bpy.data.collections.get(collection_name)["beeble_shot_uid"] = shot_uid
            bpy.data.collections.get(collection_name)["shot_with_beeble_app"] = (
                use_iphone_video
            )
            for obj in bpy.data.collections.get(collection_name).objects:
                obj["beeble_shot_uid"] = shot_uid
                obj.property_overridable_library_set('["beeble_shot_uid"]', True)
            logger.info("execute 6")
            # Store direct references in shot_attributes

            if use_iphone_video:
                new_shot.attributes.lens_anim = collection["lens"].animation_data.action
                new_shot.attributes.camera_anim = collection[
                    "camera_anchor"
                ].animation_data.action
                new_shot.attributes.footage_anim = collection[
                    "footage_anchor"
                ].animation_data.action
                new_shot.attributes.focus_anim = collection[
                    "focus"
                ].animation_data.action

            new_shot.attributes.lens = collection["lens"]
            new_shot.attributes.camera_anchor = collection["camera_anchor"]
            new_shot.attributes.footage_anchor = collection["footage_anchor"]
            new_shot.attributes.material = collection["material"]
            new_shot.attributes.camera = collection["camera"]
            new_shot.attributes.footage = collection["footage"]
            new_shot.attributes.focus = collection["focus"]

            new_shot.attributes.delete_geometry_modifier = (
                f"DeleteBackground.{collection_suffix}"
            )
            new_shot.attributes.displace_geometry_modifier = (
                f"VertexDisplacement.{collection_suffix}"
            )
            logger.info("execute 7")
            if not use_video_atlas:
                new_shot.attributes.depth_map = bpy.data.images.get(
                    f"INPUT_DEPTH_map.{collection_suffix}"
                )
                new_shot.attributes.source_map = bpy.data.images.get(
                    f"INPUT_VIDEO_map.{collection_suffix}"
                )
                new_shot.attributes.basecolor_map = bpy.data.images.get(
                    f"INPUT_BASECOLOR_map.{collection_suffix}"
                )
                new_shot.attributes.normal_map = bpy.data.images.get(
                    f"INPUT_NORMAL_map.{collection_suffix}"
                )
                new_shot.attributes.roughness_map = bpy.data.images.get(
                    f"INPUT_ROUGHNESS_map.{collection_suffix}"
                )
                new_shot.attributes.specular_map = bpy.data.images.get(
                    f"INPUT_REFLECTIVITY_map.{collection_suffix}"
                )
                new_shot.attributes.alpha_map = bpy.data.images.get(
                    f"INPUT_ALPHA_map.{collection_suffix}"
                )
            logger.info("execute 8")

            # Check camera_given and set depth settings accordingly
            if (
                camera_data.get("frames")
                and len(camera_data.get("frames")) > 0
                and camera_data.get("frames")[0].get("camera_given") is True
            ):
                # Set 2d_to_3d value to 0 on the footage object
                footage_obj = collection["footage"]
                if footage_obj:
                    footage_obj["2d_to_3d"] = 1.0

            # Set use_depth to True
            new_shot.use_depth = True

            # Open imported shot panel
            context.scene.beeble_shot_panel.show_imported_shots = True

            # Update frame end for shots if needed
            if context.scene.beeble_shot_panel.frame_end < context.scene.frame_end:
                context.scene.beeble_shot_panel.frame_end = context.scene.frame_end

            # Force scene update
            collection["footage"].modifiers.update()
            collection["camera_anchor"].update_tag()
            collection["footage_anchor"].update_tag()
            collection["footage"].update_tag()
            context.view_layer.update()

            success_message = "Shot imported successfully"
            if using_default_camera:
                success_message += " (using default camera settings)"

            logger.info(success_message)

            # Apply FPS and resolution settings based on user choices or defaults
            use_imported_fps = context.scene.get("beeble_temp_use_imported_fps", True)
            use_imported_resolution = context.scene.get(
                "beeble_temp_use_imported_resolution", True
            )

            # Set FPS based on user choice, but skip for image shots
            if use_imported_fps and shot_type != "image":
                context.scene.render.fps = int(camera_data.get("frame_rate"))

            # Set resolution based on user choice
            if use_imported_resolution:
                context.scene.render.resolution_x = camera_data.get("resolution")[0]
                context.scene.render.resolution_y = camera_data.get("resolution")[1]

            # Find the closest aspect ratio
            aspect_ratios = {
                "21:9": 9 / 21,
                "16:9": 9 / 16,
                "4:3": 3 / 4,
                "1:1": 1,
                "2:3": 3 / 2,
                "9:16": 16 / 9,
            }
            scene_aspect_ratio = new_shot.resolution_y / new_shot.resolution_x

            difference = 10
            ar_idx = None
            for key, value in aspect_ratios.items():
                if abs(scene_aspect_ratio - value) < difference:
                    difference = abs(scene_aspect_ratio - value)
                    ar_idx = key

            new_shot.aspect_ratio = ar_idx
            context.scene.view_layers.update()

            # Find the closest frame rate
            frame_rates = {
                "24": 24,
                "25": 25,
                "30": 30,
                "50": 50,
                "60": 60,
                "120": 120,
                "240": 240,
            }
            scene_fps = camera_data.get("frame_rate")

            fps_idx = None
            for key, value in frame_rates.items():
                if scene_fps == value:
                    fps_idx = key

            if fps_idx is None:
                new_shot.fps = "custom"
                new_shot.custom_fps = round(scene_fps)
            else:
                new_shot.fps = fps_idx

            context.scene.view_layers.update()
            context.scene.account_settings.active_tab = "SHOTS"
            context.scene.beeble_shot_panel.selected_shot = shot_uid

            # Set the collection name
            new_shot.collection.name = os.path.basename(self.shot_path)

            # Deselect all objects
            bpy.ops.object.select_all(action="DESELECT")

            # Select all object in new_shot.collection
            for obj in new_shot.collection.objects:
                obj.select_set(True)

            self.set_camera_view(context, new_shot.collection)

            # Track event
            get_api_client().track_event(
                "blender_import",
                {
                    "item_type": "shot",
                    "set_type": new_shot.set_type,
                    "shot_type": new_shot.type,
                },
            )

            return {"FINISHED"}

        except Exception as e:
            logger.error(f"Failed to import shot: {str(e)}")
            return {"CANCELLED"}


class BEEBLE_OT_ChooseShotDirectory(Operator):
    """Import shots from a directory"""

    bl_idname = "beeble.choose_shot_directory"
    bl_label = "Choose VFX Passes"
    bl_description = "Import a Beeble VFX Passes from .zip or folder"

    # Update filter_glob to show both .beeble and .zip files.
    filter_glob: bpy.props.StringProperty(
        default="*.beeble;*.zip",
        description="Filter files to show .zip files",
        options={"HIDDEN"},
    )
    filepath: bpy.props.StringProperty(subtype="FILE_PATH")

    def _clean_extracted_directory(self, target_dir):
        """Clean up extracted directory by removing metadata and handling redundant folders."""

        def is_metadata_directory(dirname):
            """Check if directory is a known metadata directory."""
            metadata_dirs = {
                "__MACOSX",  # MacOS metadata
                ".DS_Store",  # MacOS metadata
                "Thumbs.db",  # Windows thumbnail cache
                ".git",  # Git metadata
                ".svn",  # SVN metadata
                "__pycache__",  # Python cache
                ".idea",  # IntelliJ IDEA metadata
                ".vscode",  # VSCode metadata
            }
            return dirname in metadata_dirs

        # Handle redundant folder and clean up metadata
        contents = os.listdir(target_dir)
        real_dirs = [
            d
            for d in contents
            if os.path.isdir(os.path.join(target_dir, d))
            and not is_metadata_directory(d)
        ]

        # If there's exactly one non-metadata directory, move contents up
        # but DO NOT flatten if that directory itself is an expected pass directory
        if len(real_dirs) == 1 and real_dirs[0] != "depth":
            intermediate_dir = os.path.join(target_dir, real_dirs[0])
            for item in os.listdir(intermediate_dir):
                source_path = os.path.join(intermediate_dir, item)
                destination_path = os.path.join(target_dir, item)
                # Handle case where target already exists (e.g. after moving some items)
                if os.path.exists(destination_path):
                    if os.path.isdir(destination_path):
                        shutil.rmtree(destination_path)
                    else:
                        os.remove(destination_path)
                shutil.move(source_path, destination_path)
            os.rmdir(intermediate_dir)

        # Clean up metadata directories/files from the top level of target_dir
        # Re-list contents after potential move operation
        current_contents = os.listdir(target_dir)
        for item in current_contents:
            if is_metadata_directory(item):
                item_path = os.path.join(target_dir, item)
                if os.path.isdir(item_path):
                    shutil.rmtree(item_path)
                else:
                    os.remove(item_path)

    def execute(self, context):
        shot_path_to_use = ""

        if self.filepath.lower().endswith((".beeble", ".zip")):
            zip_file_directory = os.path.dirname(self.filepath)
            base_name_no_ext = os.path.splitext(os.path.basename(self.filepath))[0]

            # Initial target_dir is a new folder named after the zip, next to the zip file
            target_dir_base = os.path.join(zip_file_directory, base_name_no_ext)
            target_dir = target_dir_base

            # Handle name clashes for the extraction folder
            count = 1
            while os.path.exists(target_dir):
                target_dir = f"{target_dir_base}_{count}"
                count += 1

            try:
                os.makedirs(target_dir, exist_ok=True)
                with zipfile.ZipFile(self.filepath, "r") as zip_ref:
                    zip_ref.extractall(target_dir)

                self._clean_extracted_directory(target_dir)
                shot_path_to_use = target_dir
            except Exception as e:
                logger.error(f"Failed to extract or clean shot archive: {str(e)}")
                self.report({"ERROR"}, f"Failed to extract archive: {str(e)}")
                # Attempt to clean up partially created directory if extraction failed
                if os.path.exists(target_dir) and not os.listdir(
                    target_dir
                ):  # if empty
                    os.rmdir(target_dir)
                elif os.path.exists(target_dir):  # if not empty but failed
                    shutil.rmtree(
                        target_dir
                    )  # consider removing if partial extraction is an issue

                return {"CANCELLED"}

        else:  # Input is expected to be a directory
            if not os.path.isdir(self.filepath):
                bpy.ops.beeble.error_alert(
                    "INVOKE_DEFAULT",
                    message="Invalid selection! Please select either a .zip file or a directory.",
                )
                return {"CANCELLED"}
            shot_path_to_use = self.filepath  # Use the original directory path directly

        if not shot_path_to_use:
            self.report({"ERROR"}, "Shot path could not be determined.")
            return {"CANCELLED"}

        bpy.ops.beeble.import_shot(shot_path=bpy.path.abspath(shot_path_to_use))
        return {"FINISHED"}

    def invoke(self, context, event):
        # Clear the filename to prevent previous .zip basename from persisting
        self.filepath = ""
        context.window_manager.fileselect_add(self)
        return {"RUNNING_MODAL"}


class BEEBLE_OT_ActiveShot(Operator):
    """Operator to set the active shot"""

    bl_idname = "beeble.active_shot"
    bl_label = "Set Active Shot"
    bl_description = "Set the active shot to the selected shot"

    uid: bpy.props.StringProperty()

    def execute(self, context):
        bpy.ops.object.select_all(action="DESELECT")

        # Find the collection with given UID
        shot_collection = None
        for collection in context.scene.collection.children:
            if (
                "beeble_shot_uid" in collection.keys()
                and collection["beeble_shot_uid"] == self.uid
            ):
                shot_collection = collection
                break

        # Select all object in shot_collection
        if shot_collection:
            for obj in shot_collection.objects:
                obj.select_set(True)

        return {"FINISHED"}


class BEEBLE_OT_ShotExportSettings(Operator):
    """Show export settings in a modal"""

    bl_idname = "beeble.shot_export_settings"
    bl_label = "Render"
    bl_description = "Render the selected VFX Passes"

    def execute(self, context):
        scene = context.scene
        shot_item = next(
            (
                shot_item
                for shot_item in scene.beeble_imported_shots
                if shot_item.uid == scene.beeble_shot_panel.selected_shot
            ),
            None,
        )

        # Change to viewport shading before rendering
        bpy.context.space_data.shading.type = "SOLID"

        # Start the render
        if shot_item.type == "image":
            bpy.ops.render.render("INVOKE_DEFAULT", write_still=True)
        else:
            bpy.ops.render.render("INVOKE_DEFAULT", animation=True)

        # Track event
        get_api_client().track_event(
            "blender_render",
            {"set_type": shot_item.set_type, "render_engine": scene.render.engine},
        )

        return {"FINISHED"}


class BEEBLE_OT_ImportSettingsConflictDialog(Operator):
    """Dialog to handle FPS and resolution conflicts during import"""

    bl_idname = "beeble.import_settings_conflict_dialog"
    bl_label = "Import Settings Conflict"
    bl_description = "Choose which settings to use when importing VFX passes"

    # Properties to store the conflicting values
    current_fps: bpy.props.IntProperty(name="Current FPS")
    imported_fps: bpy.props.IntProperty(name="Imported FPS")
    current_resolution_x: bpy.props.IntProperty(name="Current Resolution X")
    current_resolution_y: bpy.props.IntProperty(name="Current Resolution Y")
    imported_resolution_x: bpy.props.IntProperty(name="Imported Resolution X")
    imported_resolution_y: bpy.props.IntProperty(name="Imported Resolution Y")

    # User choices
    use_imported_fps: bpy.props.BoolProperty(name="Use Imported FPS", default=True)
    use_imported_resolution: bpy.props.BoolProperty(
        name="Use Imported Resolution", default=True
    )

    # Callback properties
    shot_path: bpy.props.StringProperty()
    shot_type: bpy.props.StringProperty()  # Shot type (image or video)

    def draw(self, context):
        layout = self.layout

        # Header
        layout.label(text="Choose which settings you'd like to use.", icon="ERROR")
        layout.separator()

        # FPS Conflict Section - only show for video shots
        if self.shot_type != "image" and self.current_fps != self.imported_fps:
            fps_box = layout.box()
            fps_box.label(text="Frame Rate (FPS):", icon="TIME")
            row = fps_box.row()
            row.label(text=f"• Current Scene: {self.current_fps} FPS")
            row = fps_box.row()
            row.label(text=f"• Imported VFX Passes: {self.imported_fps} FPS")
            fps_box.separator()
            fps_box.prop(
                self, "use_imported_fps", text="Use FPS from imported VFX passes"
            )
            if self.use_imported_fps:
                fps_box.label(
                    text="Your scene will update to match the imported frame rate."
                )
            else:
                fps_box.label(text="Your scene frame rate will remain unchanged.")

        # Resolution Conflict Section
        if (
            self.current_resolution_x != self.imported_resolution_x
            or self.current_resolution_y != self.imported_resolution_y
        ):
            res_box = layout.box()
            res_box.label(text="Resolution:", icon="FULLSCREEN_ENTER")
            row = res_box.row()
            row.label(
                text=f"• Current Scene: {self.current_resolution_x} × {self.current_resolution_y}"
            )
            row = res_box.row()
            row.label(
                text=f"• Imported VFX Passes: {self.imported_resolution_x} × {self.imported_resolution_y}"
            )
            res_box.separator()
            res_box.prop(
                self,
                "use_imported_resolution",
                text="Use resolution from imported VFX passes",
            )
            if self.use_imported_resolution:
                res_box.label(
                    text="Your scene will update to match the imported resolution."
                )
            else:
                res_box.label(text="Your scene resolution will remain unchanged.")

    def execute(self, context):
        # Store the user's choices for the import operation to use
        # For image shots, always set use_imported_fps to False to prevent FPS overwrite
        if self.shot_type == "image":
            context.scene["beeble_temp_use_imported_fps"] = False
        else:
            context.scene["beeble_temp_use_imported_fps"] = self.use_imported_fps

        context.scene["beeble_temp_use_imported_resolution"] = (
            self.use_imported_resolution
        )

        # Execute the import with user's choices
        bpy.ops.beeble.import_shot(shot_path=self.shot_path)

        # Clean up temporary properties
        if "beeble_temp_use_imported_fps" in context.scene:
            del context.scene["beeble_temp_use_imported_fps"]
        if "beeble_temp_use_imported_resolution" in context.scene:
            del context.scene["beeble_temp_use_imported_resolution"]

        return {"FINISHED"}

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self, width=400)
