import bpy
from bpy.types import Operator

from ...core.logger import logger


class BEEBLE_OT_EditShot(Operator):
    bl_idname = "beeble.edit_shot"
    bl_label = "Edit Shot"
    bl_description = "Edit shot details"
    shot_uid: bpy.props.StringProperty()

    def set_frame_range(self, context, collection):
        # Get frame ranges from custom properties
        viewport_start = collection["frame_start"]
        viewport_end = collection["frame_end"]

        # Set viewport (preview) frame range
        context.scene.frame_start = viewport_start
        context.scene.frame_end = viewport_end

        # Set current frame to start frame
        context.scene.frame_current = viewport_start

    def set_camera_view(self, context, collection):
        # Find the camera in the collection
        shot_camera = None
        for obj in collection.objects:
            if obj.type == 'CAMERA' and "BeebleCam." in obj.name:
                shot_camera = obj
                break
        if shot_camera:
            # Set the active camera
            context.scene.camera = shot_camera

            # Change viewport to camera view
            for area in context.screen.areas:
                if area.type == 'VIEW_3D':
                    for space in area.spaces:
                        if space.type == 'VIEW_3D':
                            space.region_3d.view_perspective = 'CAMERA'
                            break

    def set_perspective_view(self, context):
        for area in context.screen.areas:
            if area.type == 'VIEW_3D':
                for space in area.spaces:
                    if space.type == 'VIEW_3D':
                        space.region_3d.view_perspective = 'PERSP'
                        break

    def execute(self, context):
        logger.info(f"Editing shot {self.shot_uid}")
        shot_item = next((shot_item for shot_item in context.scene.beeble_imported_shots
                           if shot_item.uid == self.shot_uid), None)
        collection = shot_item.collection

        if context.scene.beeble_shot_panel.selected_shot == self.shot_uid:
            context.scene.beeble_shot_panel.selected_shot = ""  # Deselect if already selected
            self.set_perspective_view(context)
            return {'FINISHED'}
        else: # Click the button
            context.scene.beeble_shot_panel.selected_shot = self.shot_uid  # Select new shot
            self.set_camera_view(context, collection)

        if collection:
            # Set frame ranges
            self.set_frame_range(context, collection)

            logger.info(f"Shot {self.shot_uid} settings applied successfully")
            self.report({'INFO'}, f"Shot {self.shot_uid} settings applied successfully")
        else:
            bpy.ops.beeble.error_alert('INVOKE_DEFAULT', message="Shot collection not found")
            return {'CANCELLED'}

        return {'FINISHED'}

class BEEBLE_OT_ToogleShotVisibility(Operator):
    bl_idname = "beeble.toggle_shot_visibility"
    bl_label = "Toggle Shot Visibility"
    bl_description = "Toggle shot visibility"

    shot_uid: bpy.props.StringProperty()

    @classmethod
    def description(cls, context, properties):
        if properties.shot_uid:
            # Find the shot item using the provided shot_uid
            shot_item = next((shot_item for shot_item in context.scene.beeble_imported_shots
                              if shot_item.uid == properties.shot_uid), None)
            if shot_item and shot_item.collection:
                # Check the visibility state
                layer_collection = context.view_layer.layer_collection.children.get(shot_item.collection.name)
                if layer_collection:
                    if layer_collection.hide_viewport:
                        return "Show this shot"
                    else:
                        return "Hide this shot"

        return "Toggle shot visibility"

    def execute(self, context):
        shot_item = next((shot_item for shot_item in context.scene.beeble_imported_shots
                           if shot_item.uid == self.shot_uid), None)
        shot_collection = shot_item.collection
        if shot_collection:
            # Get the current visibility state from the first view layer
            layer_collection = context.view_layer.layer_collection.children.get(shot_collection.name)
            if layer_collection:
                # Toggle visibility based on current viewport state
                is_visible = layer_collection.hide_viewport
                self.toggle_collection_visibility(layer_collection, not is_visible)
                # Also toggle render visibility in the collection itself
                self.toggle_render_visibility(shot_collection, not is_visible)
                # Toggle object visibility in viewport and render
                self.toggle_objects_visibility(shot_collection, not is_visible)

        return {'FINISHED'}

    def toggle_collection_visibility(self, layer_collection, hide):
        """Recursively toggle viewport visibility of collection and its children"""
        layer_collection.hide_viewport = hide
        # Toggle all child collections
        for child in layer_collection.children:
            self.toggle_collection_visibility(child, hide)

    def toggle_render_visibility(self, collection, hide):
        """Recursively toggle render visibility of collection and its children"""
        collection.hide_render = hide
        # Toggle all child collections
        for child in collection.children:
            self.toggle_render_visibility(child, hide)

    def toggle_objects_visibility(self, collection, hide):
        """Recursively toggle visibility of all objects in collection and its children"""
        # Toggle objects in current collection
        for obj in collection.objects:
            obj.hide_set(hide)
            obj.hide_render = hide
            # If the object is an instance collection, toggle its objects too
            if obj.instance_type == 'COLLECTION' and obj.instance_collection:
                self.toggle_objects_visibility(obj.instance_collection, hide)

        # Toggle objects in child collections
        for child in collection.children:
            self.toggle_objects_visibility(child, hide)

class BEEBLE_OT_CloseShotDetails(bpy.types.Operator):
    bl_idname = "beeble.close_shot_details"
    bl_label = "Close Shot Details"

    def set_frame_range(self, context):
        # Set viewport (preview) frame range
        context.scene.frame_start = context.scene.beeble_shot_panel.frame_start
        context.scene.frame_end = context.scene.beeble_shot_panel.frame_end

    def set_view(self, context):
        # Change viewport to camera view
        for area in context.screen.areas:
            if area.type == 'VIEW_3D':
                for space in area.spaces:
                    if space.type == 'VIEW_3D':
                        # Check if currently in camera view
                        if space.region_3d.view_perspective == 'CAMERA':
                            # Reset to default perspective view
                            space.region_3d.view_perspective = 'PERSP'
                            break

    def execute(self, context):
        shot_panel_props = context.scene.beeble_shot_panel
        shot_panel_props.selected_shot = ""

        if shot_panel_props.show_imported_shots:
            # Remove that shot from imported shots
            for idx, shot_item in enumerate(context.scene.beeble_imported_shots):
                shot_exists = any(obj for obj in context.scene.objects
                                if "beeble_shot_uid" in obj.keys() and obj["beeble_shot_uid"] == shot_item.uid)
                if not shot_exists:
                    context.scene.beeble_imported_shots.remove(idx)

        # Set frame ranges
        self.set_frame_range(context)

        # Set view
        self.set_view(context)

        return {'FINISHED'}


class BEEBLE_OT_ClearResolution(Operator):
    bl_idname = "beeble.clear_resolution"
    bl_label = "Clear Resolution"
    bl_description = "Clear resolution to shot default"

    def execute(self, context):
        scene = context.scene
        shot_item = next((shot for shot in scene.beeble_imported_shots
                         if shot.uid == scene.beeble_shot_panel.selected_shot), None)

        if not shot_item:
            return {'CANCELLED'}

        shot_item.resolution_y = shot_item.original_resolution_y

        self.report({'INFO'}, "Resolution reset to scene defaults")
        return {'FINISHED'}

class BEEBLE_OT_ClearFrame(Operator):
    bl_idname = "beeble.clear_frame"
    bl_label = "Clear Frame"
    bl_description = "Clear start/end frame to shot default"

    def execute(self, context):
        scene = context.scene
        shot_item = next((shot for shot in scene.beeble_imported_shots
                         if shot.uid == scene.beeble_shot_panel.selected_shot), None)

        if not shot_item:
            return {'CANCELLED'}

        scene.frame_start = 0
        scene.frame_end = shot_item.frame_count - 1

        self.report({'INFO'}, "Start/end frame reset to scene defaults")
        return {'FINISHED'}
