# operators/account_operators.py
import os
import shutil
import webbrowser

import bpy
from bpy.types import Operator


from ...core.api import get_api_client
from ...core.session import AuthManager, LoginStatus
from ...core.constants import AUTH_SIGNIN_URL
from ...core.logger import logger
from ...core.version_manager import check_for_updates, download_addon_update


class BEEBLE_OT_BrowserLogin(Operator):
    bl_idname = "login.browser_login"
    bl_label = "Login with Browser"
    bl_description = "Login to use Beeble Add-on"

    _timer = None
    _poll_count = 0
    MAX_POLLS = 300  # Maximum number of polling attempts (5 minutes with 1s interval)

    # Keep track of active instance
    _active_instance = None

    @classmethod
    def cleanup_login(cls, context):
        """Static method to cleanup login state and timer"""
        try:
            # If we have an active instance, clean it up
            if cls._active_instance is not None:
                if (
                    hasattr(cls._active_instance, "_timer")
                    and cls._active_instance._timer is not None
                ):
                    context.window_manager.event_timer_remove(
                        cls._active_instance._timer
                    )
                    cls._active_instance._timer = None
                cls._active_instance = None
        except ReferenceError:
            # Handle case where the operator instance was already removed
            cls._active_instance = None

        # Reset login state
        context.scene.account_settings.login_state = "LOGGED_OUT"
        context.scene.account_settings.status_message = "Login cancelled"

    def modal(self, context, event):
        # Check if this instance is still active
        if self != self.__class__._active_instance:
            logger.warning("Inactive instance detected, cleaning up")
            self.cleanup(context)
            return {"CANCELLED"}

        if event.type == "TIMER":
            self._poll_count += 1

            if self._poll_count > self.MAX_POLLS:
                context.scene.account_settings.status_message = (
                    "Login timeout. Please try again."
                )
                context.scene.account_settings.login_state = "LOGGED_OUT"
                self.cleanup(context)
                return {"CANCELLED"}

            try:
                auth_manager = AuthManager.get_instance()
                status, success = auth_manager.check_browser_login_status(context)

                if status == LoginStatus.ERROR:
                    self._redraw_view3d(context)
                    self.cleanup(context)
                    return {"CANCELLED"}

                if status == LoginStatus.AUTHENTICATED and success:
                    self._redraw_view3d(context)
                    self.cleanup(context)
                    return {"FINISHED"}

                return {"PASS_THROUGH"}

            except Exception as e:
                context.scene.account_settings.login_state = "LOGGED_OUT"
                context.scene.account_settings.status_message = str(e)
                self.cleanup(context)
                return {"FINISHED"}

        return {"PASS_THROUGH"}

    def execute(self, context):
        try:
            # Start logger
            logger.setup_logger(context.scene.account_settings.workspace_path)

            # Cancel any existing login attempt
            if self.__class__._active_instance is not None:
                self.__class__.cleanup_login(context)

            # Set this as the active instance
            self.__class__._active_instance = self

            # Set initial state
            context.scene.account_settings.login_state = "LOGGING_IN"
            context.scene.account_settings.status_message = "Initializing login..."

            api_client = get_api_client()
            blender_token = api_client.issue_blender_token()

            if not blender_token:
                context.scene.account_settings.login_state = "LOGGED_OUT"
                context.scene.account_settings.status_message = (
                    "Failed to get Blender token"
                )
                return {"CANCELLED"}

            api_client.set_blender_token(blender_token)
            webbrowser.open(f"{AUTH_SIGNIN_URL}?token={blender_token}")

            wm = context.window_manager
            self._timer = wm.event_timer_add(1, window=context.window)
            wm.modal_handler_add(self)

            context.scene.account_settings.status_message = (
                "Please complete login in your browser..."
            )
            return {"RUNNING_MODAL"}

        except Exception as e:
            context.scene.account_settings.login_state = "LOGGED_OUT"
            context.scene.account_settings.status_message = str(e)
            return {"CANCELLED"}

    @staticmethod
    def _redraw_view3d(context):
        for area in context.screen.areas:
            if area.type == "VIEW_3D":
                area.tag_redraw()

    def cleanup(self, context):
        """Clean up timer"""
        if self._timer:
            context.window_manager.event_timer_remove(self._timer)
            self._timer = None


class BEEBLE_OT_CancelLogin(Operator):
    bl_idname = "login.cancel_login"
    bl_label = "Cancel Login"
    bl_description = "Cancel the current login attempt"

    def execute(self, context):
        # Use the static cleanup method from BEEBLE_OT_BrowserLogin
        BEEBLE_OT_BrowserLogin.cleanup_login(context)
        try:
            # First update DynamoDB status
            success = get_api_client().cancel_login()
            if not success:
                logger.warning("Warning: Failed to update DynamoDB status on logout")

        except Exception as e:
            logger.warning(f"Warning: Error during cancel: {str(e)}")

        return {"FINISHED"}


class BEEBLE_OT_Logout(Operator):
    bl_idname = "login.logout"
    bl_label = "Logout"
    bl_description = "Logout from the current session"

    def execute(self, context):
        BEEBLE_OT_BrowserLogin.cleanup_login(context)

        session_manager = AuthManager.get_instance()
        session_manager.logout(context)

        return {"FINISHED"}


class BEEBLE_OT_EmptyWorkspace(Operator):
    """Empty temporary workspace directories"""

    bl_idname = "login.empty_workspace"
    bl_label = "Empty Workspace"
    bl_description = "Clean up the addon and restart Blender"
    bl_options = {"REGISTER", "UNDO"}

    def empty_directories(self, context):
        """Empty temporary directory used for previews"""
        try:
            if (
                not hasattr(context.scene, "account_settings")
                or not context.scene.account_settings.workspace_path
            ):
                return False

            workspace_path = bpy.path.abspath(
                context.scene.account_settings.workspace_path
            )
            if not workspace_path:
                return False

            # Define directories to empty
            directories = [
                os.path.join(workspace_path, "Temp"),
                os.path.join(workspace_path, "Sets"),
                os.path.join(workspace_path, "Shots"),
                os.path.join(workspace_path, "Worlds"),
                os.path.join(workspace_path, "Interiors"),
            ]

            # Empty each directory
            for directory in directories:
                if os.path.exists(directory):
                    shutil.rmtree(directory)

            # Empty settings.json
            if os.path.exists(os.path.join(workspace_path, "settings.json")):
                os.remove(os.path.join(workspace_path, "settings.json"))

            return True

        except Exception as e:
            logger.error(f"Error emptying directories: {str(e)}")
            return False

    def execute(self, context):
        success = self.empty_directories(context)

        if success:
            # Set update state to CACHE_CLEANUP_COMPLETED to show the quit button in cache section
            context.scene.account_settings.update_state = "CACHE_CLEANUP_COMPLETED"
            context.scene.account_settings.update_status = (
                "Cleanup completed successfully.\nPlease quit Blender to complete."
            )
            self.report({"INFO"}, "Workspace emptied successfully")

            # Force UI redraw to show the quit button
            for area in context.screen.areas:
                if area.type == "VIEW_3D":
                    area.tag_redraw()
        else:
            self.report({"ERROR"}, "Failed to empty workspace")

        return {"FINISHED"}


class BEEBLE_OT_CheckForUpdates(Operator):
    """Check for addon updates"""

    bl_idname = "beeble.check_for_updates"
    bl_label = "Check for Updates"
    bl_description = "Check for addon updates"

    def execute(self, context):
        settings = context.scene.account_settings

        try:
            # Check for updates
            update_available, update_data = check_for_updates()

            # Update UI
            settings.update_available = update_available
            if update_data:
                settings.update_status = update_data.get("message", "")
                settings.latest_addon_version = update_data.get(
                    "addon_version", "0.0.0"
                )
            else:
                settings.update_status = "No update information available"
            settings.update_state = "NONE"

            if update_available:
                self.report(
                    {"INFO"}, f"Update available: {settings.latest_addon_version}"
                )
                logger.info(f"Update available: {settings.latest_addon_version}")
            else:
                self.report({"INFO"}, "No updates available")
                logger.info("No updates available")

            # Force UI redraw
            for area in context.screen.areas:
                if area.type == "VIEW_3D":
                    area.tag_redraw()

            return {"FINISHED"}

        except Exception as e:
            logger.error(f"Error checking for updates: {str(e)}")
            settings.update_status = f"Error: {str(e)}"
            settings.update_state = "FAILED"
            self.report({"ERROR"}, f"Error checking for updates: {str(e)}")
            return {"CANCELLED"}


class BEEBLE_OT_DownloadUpdate(Operator):
    """Download and install addon update"""

    bl_idname = "beeble.download_update"
    bl_label = "Download & Install Update"
    bl_description = "Download and install the latest addon update"

    _timer = None

    def modal(self, context, event):
        settings = context.scene.account_settings

        if event.type == "TIMER":
            # Check if download/install is complete
            if (
                settings.update_state == "COMPLETED"
                or settings.update_state == "FAILED"
            ):
                self.cancel(context)
                return {"FINISHED"}

            # Force UI redraw
            for area in context.screen.areas:
                if area.type == "VIEW_3D":
                    area.tag_redraw()

        return {"PASS_THROUGH"}

    def execute(self, context):
        settings = context.scene.account_settings

        # Reset progress
        settings.update_download_progress = 0.0
        settings.update_status = "Starting download and installation..."
        settings.update_state = "DOWNLOADING"

        # Set up timer for UI updates
        wm = context.window_manager
        self._timer = wm.event_timer_add(0.1, window=context.window)
        wm.modal_handler_add(self)

        # Progress callback for download and install
        def progress_callback(progress, status):
            settings.update_download_progress = progress
            settings.update_status = status
            logger.info(f"Update progress: {progress:.2f} - {status}")

            # Update state based on status message
            if progress >= 1.0 and "complete" in status.lower():
                settings.update_state = "COMPLETED"
            elif "installing" in status.lower():
                settings.update_state = "INSTALLING"
            elif "error" in status.lower():
                settings.update_state = "FAILED"

        # Start download and installation
        try:
            success, message, zip_path = download_addon_update(
                context, progress_callback
            )

            if not success:
                settings.update_state = "FAILED"
                settings.update_status = message
                logger.error(f"Update failed: {message}")
                self.report({"ERROR"}, f"Update failed: {message}")
                return {"CANCELLED"}

            # Store the downloaded path
            settings.downloaded_addon_path = zip_path
            logger.info(f"Download and installation started: {zip_path}")

            return {"RUNNING_MODAL"}

        except Exception as e:
            settings.update_state = "FAILED"
            settings.update_status = f"Error: {str(e)}"
            logger.error(f"Error starting update: {str(e)}")
            self.report({"ERROR"}, f"Error starting update: {str(e)}")
            return {"CANCELLED"}

    def cancel(self, context):
        if self._timer:
            context.window_manager.event_timer_remove(self._timer)
            self._timer = None


class BEEBLE_OT_QuitBlender(Operator):
    """Quit Blender after addon update"""

    bl_idname = "beeble.quit_blender"
    bl_label = "Quit Blender"
    bl_description = "Quit Blender to apply the addon update"

    def execute(self, context):
        # Save user preferences to make the addon installation permanent
        bpy.ops.wm.save_userpref()

        # Show a message to the user
        self.report(
            {"INFO"}, "Blender will close. Please restart it to use the updated addon."
        )
        logger.info("Scheduling Blender quit...")

        # Schedule quit after a short delay
        def quit_blender():
            bpy.ops.wm.quit_blender()
            return None

        # Schedule quit after a short delay
        bpy.app.timers.register(quit_blender, first_interval=0.1)

        return {"FINISHED"}
