import os
import bpy
import json
import urllib.request
import urllib.error
import zipfile
import tempfile
import shutil
import threading

from .logger import logger
from .constants import LATEST_JSON_URL, LATEST_ZIP_URL


def get_current_version():
    """Get the current version from bl_info"""
    from .. import bl_info

    return ".".join(map(str, bl_info["version"]))


def parse_version(version_string):
    """Parse version string into tuple for comparison"""
    if isinstance(version_string, tuple):
        return version_string

    # Handle tuple-style version strings like '(1, 0, 0)'
    if version_string.startswith("(") and version_string.endswith(")"):
        version_string = version_string.strip("()").replace(" ", "")

    return tuple(map(int, version_string.split(".")))


def compare_versions(v1, v2):
    """Compare two version strings
    Returns: -1 if v1 < v2, 0 if v1 == v2, 1 if v1 > v2
    """
    v1_tuple = parse_version(v1)
    v2_tuple = parse_version(v2)

    if v1_tuple < v2_tuple:
        return -1
    elif v1_tuple > v2_tuple:
        return 1
    else:
        return 0


def get_addon_path():
    """Get the path to the addon"""
    return os.path.dirname(os.path.dirname(os.path.realpath(__file__)))


def get_template_path():
    """Get the path to the bundled template.blend file

    Returns:
        str: Path to template.blend in the addon directory
    """
    addon_path = get_addon_path()
    return os.path.join(addon_path, "template.blend")


def get_blender_addons_path():
    """Get the path to Blender's addons directory"""
    addon_directory = bpy.utils.user_resource("SCRIPTS", path="addons")
    return addon_directory


def check_for_updates():
    """Check if updates are available by fetching latest.json

    Returns:
        (update_needed, update_data) tuple where:
        - update_needed is a boolean indicating if an update is needed
        - update_data is the full update data dictionary or None if error
    """
    try:
        current_version = get_current_version()
        update_data = fetch_latest_json()

        if not update_data:
            return False, None

        # latest.json uses "version" field
        latest_version = update_data.get("version", "0.0.0")

        # Check if current version is below minimum supported version
        min_supported_version = update_data.get("min_supported_version", "0.0.0")
        if (
            min_supported_version
            and compare_versions(current_version, min_supported_version) < 0
        ):
            update_data["block_addon"] = True
        else:
            update_data["block_addon"] = False

        # Check if update is needed
        update_needed = compare_versions(current_version, latest_version) < 0

        return update_needed, update_data

    except Exception as e:
        logger.error(f"Error checking for updates: {str(e)}")
        return False, None


def fetch_latest_json():
    """
    Fetch latest.json from the release server.
    Returns: dict containing update info if successful, None otherwise
    """
    try:
        logger.info(f"Fetching latest.json from {LATEST_JSON_URL}")

        req = urllib.request.Request(
            LATEST_JSON_URL,
            headers={"User-Agent": "Mozilla/5.0 (Blender Addon Updater)"},
        )
        response = urllib.request.urlopen(req, timeout=10)
        data = json.loads(response.read().decode("utf-8"))

        # Ensure required fields have defaults
        if "version" not in data:
            data["version"] = "0.0.0"
        if "message" not in data:
            data["message"] = ""
        if "min_supported_version" not in data:
            data["min_supported_version"] = "0.0.0"

        return data

    except urllib.error.HTTPError as e:
        logger.error(f"HTTP Error fetching latest.json: {e.code} {e.reason}")
        return None
    except urllib.error.URLError as e:
        logger.error(f"URL Error fetching latest.json: {e.reason}")
        return None
    except json.JSONDecodeError as e:
        logger.error(f"Error parsing latest.json: {str(e)}")
        return None
    except Exception as e:
        logger.error(f"Error fetching latest.json: {str(e)}")
        return None


def download_addon_update(context, progress_callback=None):
    """
    Download and install addon update from latest.zip
    Args:
        context: Blender context
        progress_callback: Callback function to report progress

    Returns:
        (success, message, filepath) tuple
    """
    try:
        temp_dir = tempfile.mkdtemp()
        temp_zip_path = os.path.join(temp_dir, "addon_update.zip")

        def _download_and_install():
            try:
                logger.info(f"Downloading addon from: {LATEST_ZIP_URL}")

                req = urllib.request.Request(
                    LATEST_ZIP_URL,
                    headers={"User-Agent": "Mozilla/5.0 (Blender Addon Updater)"},
                )

                response = urllib.request.urlopen(req)
                total_size = int(response.headers.get("content-length", 0))

                if total_size == 0:
                    logger.warning(
                        "Download file size is 0, this might indicate an issue"
                    )
                    if progress_callback:
                        progress_callback(
                            0.5, "Download complete, starting installation..."
                        )
                else:
                    downloaded_size = 0
                    block_size = 8192

                    with open(temp_zip_path, "wb") as f:
                        while True:
                            buffer = response.read(block_size)
                            if not buffer:
                                break

                            f.write(buffer)
                            downloaded_size += len(buffer)

                            if progress_callback and total_size > 0:
                                progress = (downloaded_size / total_size) * 0.5
                                progress_callback(
                                    progress, f"Downloading: {int(progress * 100 * 2)}%"
                                )

                    if progress_callback:
                        progress_callback(
                            0.5, "Download complete, starting installation..."
                        )

                # Proceed with installation
                try:
                    if progress_callback:
                        progress_callback(0.6, "Extracting addon...")

                    temp_extract_dir = tempfile.mkdtemp()
                    with zipfile.ZipFile(temp_zip_path, "r") as zip_ref:
                        zip_ref.extractall(temp_extract_dir)

                    if progress_callback:
                        progress_callback(0.7, "Locating addon files...")

                    extracted_addon_dir = None
                    for root, dirs, files in os.walk(temp_extract_dir):
                        if "__init__.py" in files:
                            extracted_addon_dir = root
                            break

                    if not extracted_addon_dir:
                        error_msg = "Could not find addon directory in downloaded zip"
                        logger.error(error_msg)
                        if progress_callback:
                            progress_callback(1.0, f"Error: {error_msg}")
                        return False, error_msg, None

                    if progress_callback:
                        progress_callback(0.8, "Installing addon...")

                    addon_path = get_addon_path()
                    addon_name = os.path.basename(addon_path)
                    addons_dir = get_blender_addons_path()
                    target_dir = os.path.join(addons_dir, addon_name)

                    if os.path.exists(target_dir):
                        shutil.rmtree(target_dir)

                    os.makedirs(target_dir, exist_ok=True)

                    for item in os.listdir(extracted_addon_dir):
                        s = os.path.join(extracted_addon_dir, item)
                        d = os.path.join(target_dir, item)
                        if os.path.isdir(s):
                            shutil.copytree(s, d, dirs_exist_ok=True)
                        else:
                            shutil.copy2(s, d)

                    if progress_callback:
                        progress_callback(0.9, "Finalizing installation...")

                    bpy.ops.wm.save_userpref()

                    shutil.rmtree(temp_extract_dir)
                    shutil.rmtree(os.path.dirname(temp_zip_path))

                    if progress_callback:
                        progress_callback(
                            1.0,
                            "Installation complete!\nRestart Blender to activate the update.",
                        )

                    return True, "Installation complete", None

                except Exception as e:
                    error_msg = f"Error during installation: {str(e)}"
                    logger.error(error_msg)
                    if progress_callback:
                        progress_callback(1.0, f"Error: {error_msg}")
                    return False, error_msg, None

            except urllib.error.HTTPError as e:
                error_msg = f"HTTP Error {e.code}: {e.reason}"
                logger.error(f"Error downloading addon update: {error_msg}")
                if progress_callback:
                    progress_callback(0, f"Error: {error_msg}")
                return False, error_msg, None
            except urllib.error.URLError as e:
                error_msg = f"URL Error: {e.reason}"
                logger.error(f"Error downloading addon update: {error_msg}")
                if progress_callback:
                    progress_callback(0, f"Error: {error_msg}")
                return False, error_msg, None
            except Exception as e:
                error_msg = f"Error downloading addon update: {str(e)}"
                logger.error(error_msg)
                if progress_callback:
                    progress_callback(0, f"Error: {str(e)}")
                return False, error_msg, None

        download_thread = threading.Thread(target=_download_and_install)
        download_thread.start()

        return True, "Download and installation started", temp_zip_path

    except Exception as e:
        logger.error(f"Error starting addon download: {str(e)}")
        return False, f"Error starting addon download: {str(e)}", None
