import os
import bpy
import math
import mathutils

from .logger import logger
from .constants import (
    UNITY2BLENDER,
    IDENTITY_MATRIX,
    ROTATE_PORTRAIT,
    ORIENTATION_PORTRAIT,
    ROTATE_UPSIDE_DOWN,
    ORIENTATION_UPSIDE_DOWN,
    ROTATE_LANDSCAPE_LEFT,
    ORIENTATION_LANDSCAPE_LEFT,
    ROTATE_LANDSCAPE_RIGHT,
    ORIENTATION_LANDSCAPE_RIGHT,
)
from ..core.version_manager import get_template_path


def get_frame_format(shot_path, map_type="source", file_ext="png"):
    """
    Determine the frame number format (6-digit or 4-digit) for backward compatibility.

    Args:
        shot_path: Path to the shot directory
        map_type: Type of map (e.g., 'source', 'basecolor', 'depth', 'alpha')
        file_ext: File extension (e.g., 'png', 'exr')

    Returns:
        str: Frame format string (e.g., '/000000.png' or '/0000.png')
    """
    # Check for {map_type}_{6-digit} format first (newer version)
    maptype_six_digit_path = os.path.join(
        shot_path, f"{map_type}/{map_type}_000001.{file_ext}"
    )
    if os.path.exists(maptype_six_digit_path):
        return f"/{map_type}_000001.{file_ext}"

    # Check for 6-digit format first (newer version)
    six_digit_path = os.path.join(shot_path, f"{map_type}/000000.{file_ext}")
    if os.path.exists(six_digit_path):
        return f"/000000.{file_ext}"

    # Fall back to 4-digit format (older version)
    four_digit_path = os.path.join(shot_path, f"{map_type}/0000.{file_ext}")
    if os.path.exists(four_digit_path):
        return f"/0000.{file_ext}"

    # Default to 6-digit format if neither exists
    return f"/000000.{file_ext}"


def is_blender_version_compatible(major, minor, patch=0):
    """
    Check if current Blender version is at least the specified version
    Args:
        major, minor, patch: Version numbers to check against
    Returns:
        bool: True if current version is equal or higher than specified
    """
    return bpy.app.version >= (major, minor, patch)


def set_action_slot_if_compatible(animation_data):
    """
    Set action slot if the Blender version supports it
    Args:
        animation_data: The animation_data to set the action slot for
    """
    # action_suitable_slots should only be used with Blender 4.4 or higher
    if (
        is_blender_version_compatible(4, 4)
        and hasattr(animation_data, "action_suitable_slots")
        and animation_data.action_suitable_slots
    ):
        animation_data.action_slot = animation_data.action_suitable_slots[0]


def get_camera_orientation(camera_orientation_number):
    camera_orientation = IDENTITY_MATRIX
    if camera_orientation_number == ORIENTATION_PORTRAIT:
        camera_orientation = ROTATE_PORTRAIT
    elif camera_orientation_number == ORIENTATION_UPSIDE_DOWN:
        camera_orientation = ROTATE_UPSIDE_DOWN
    elif camera_orientation_number == ORIENTATION_LANDSCAPE_LEFT:
        camera_orientation = ROTATE_LANDSCAPE_LEFT
    elif camera_orientation_number == ORIENTATION_LANDSCAPE_RIGHT:
        camera_orientation = ROTATE_LANDSCAPE_RIGHT
    else:
        print("CAM ORIENTATION NUM: ", camera_orientation_number)
    return camera_orientation


def calculate_relative_camera_transform(camera_transform, camera_data):
    """
    Calculate relative camera transform based on studio transform and orientation.

    Args:
        camera_transform: Raw camera transform matrix
        camera_data: Dictionary containing camera data with studio transform and orientation

    Returns:
        mathutils.Matrix: Relative camera transform
    """
    studio_transform = camera_data["studio"]["transform"]
    cam_orientation_matrix = get_camera_orientation(camera_data.get("orientation"))

    camera_mat = mathutils.Matrix(camera_transform)
    studio_mat = mathutils.Matrix(studio_transform)

    camera_transform_blender = UNITY2BLENDER @ (camera_mat @ cam_orientation_matrix)
    studio_transform_blender = UNITY2BLENDER @ (studio_mat @ IDENTITY_MATRIX)

    relative_camera_transform = (
        studio_transform_blender.inverted() @ camera_transform_blender
    )

    return relative_camera_transform


def apply_pbr_maps(
    material_obj, shot_type, shot_path, camera_data, suffix, use_video_atlas
):
    """
    Apply PBR maps to the material object with fallback defaults for missing textures.

    Args:
        material_obj: Material object to apply maps to
        shot_type: Type of shot (video or image)
        shot_path: Path to shot directory
        camera_data: Dictionary containing camera data
        suffix: Suffix for the material names

    Returns:
        tuple: (bool, str) Success status and message
    """

    print(
        "Apply PBR maps / shot_path: ",
        shot_path,
        "shot_type: ",
        shot_type,
        "use_video_atlas: ",
        use_video_atlas,
    )
    try:
        if not use_video_atlas:
            mp4_exist = os.path.exists(os.path.join(shot_path, "Source.mp4"))
            if mp4_exist:
                # Define PBR map paths and their default values
                map_definitions = {
                    "INPUT_VIDEO": {
                        "path": os.path.join(shot_path, "Source.mp4"),
                        "colorspace": "sRGB",
                    },
                    "INPUT_BASECOLOR": {
                        "path": os.path.join(shot_path, "BaseColor.mp4"),
                        "colorspace": "sRGB",
                    },
                    "INPUT_NORMAL": {
                        "path": os.path.join(shot_path, "Normal.mp4"),
                        "colorspace": "Non-Color",
                    },
                    "INPUT_ROUGHNESS": {
                        "path": os.path.join(shot_path, "Roughness.mp4"),
                        "colorspace": "Non-Color",
                    },
                    "INPUT_REFLECTIVITY": {
                        "path": os.path.join(shot_path, "Specular.mp4"),
                        "colorspace": "Non-Color",
                    },
                    "INPUT_METALLIC": {
                        "path": os.path.join(shot_path, "Metallic.mp4"),
                        "colorspace": "Non-Color",
                    },
                    "INPUT_ALPHA": {
                        "path": os.path.join(shot_path, "Alpha.mp4"),
                        "colorspace": "Non-Color",
                    },
                }
            else:
                large_png_exist = os.path.exists(os.path.join(shot_path, "Source"))
                if large_png_exist:
                    # Define file extensions based on shot type - support both 6-digit and 4-digit formats
                    source_extension = get_frame_format(shot_path, "Source", "png")
                    basecolor_extension = get_frame_format(
                        shot_path, "BaseColor", "png"
                    )
                    normal_extension = get_frame_format(shot_path, "Normal", "png")
                    roughness_extension = get_frame_format(
                        shot_path, "Roughness", "png"
                    )
                    specular_extension = get_frame_format(shot_path, "Specular", "png")
                    metallic_extension = get_frame_format(shot_path, "Metallic", "png")
                    alpha_extension = get_frame_format(shot_path, "Alpha", "png")

                    # Define PBR map paths and their default values
                    map_definitions = {
                        "INPUT_VIDEO": {
                            "path": os.path.join(
                                shot_path, f"Source{source_extension}"
                            ),
                            "default_color": (0.0, 0.0, 0.0, 0.0),  # black
                            "colorspace": "sRGB",
                        },
                        "INPUT_BASECOLOR": {
                            "path": os.path.join(
                                shot_path, f"BaseColor{basecolor_extension}"
                            ),
                            "default_color": (1.0, 1.0, 1.0, 1.0),  # white
                            "colorspace": "sRGB",
                        },
                        "INPUT_NORMAL": {
                            "path": os.path.join(
                                shot_path, f"Normal{normal_extension}"
                            ),
                            "default_color": (
                                0.5,
                                0.5,
                                1.0,
                                1.0,
                            ),  # Default normal map (pointing forward)
                            "colorspace": "Non-Color",
                        },
                        "INPUT_ROUGHNESS": {
                            "path": os.path.join(
                                shot_path, f"Roughness{roughness_extension}"
                            ),
                            "default_color": (0.5, 0.5, 0.5, 1.0),  # Medium roughness
                            "colorspace": "Non-Color",
                        },
                        "INPUT_REFLECTIVITY": {
                            "path": os.path.join(
                                shot_path, f"Specular{specular_extension}"
                            ),
                            "default_color": (
                                0.5,
                                0.5,
                                0.5,
                                1.0,
                            ),  # Default dielectric specular
                            "colorspace": "Non-Color",
                        },
                        "INPUT_METALLIC": {
                            "path": os.path.join(
                                shot_path, f"Metallic{metallic_extension}"
                            ),
                            "default_color": (0.0, 0.0, 0.0, 1.0),  # Default metallic
                            "colorspace": "Non-Color",
                        },
                        "INPUT_ALPHA": {
                            "path": os.path.join(shot_path, f"Alpha{alpha_extension}"),
                            "default_color": (1.0, 1.0, 1.0, 1.0),  # Fully opaque
                            "colorspace": "Non-Color",
                        },
                    }
                else:
                    # Define file extensions based on shot type - support both 6-digit and 4-digit formats
                    extension = get_frame_format(shot_path, "source", "png")

                    # Define PBR map paths and their default values
                    map_definitions = {
                        "INPUT_VIDEO": {
                            "path": os.path.join(shot_path, f"source{extension}"),
                            "default_color": (0.0, 0.0, 0.0, 0.0),  # black
                            "colorspace": "sRGB",
                        },
                        "INPUT_BASECOLOR": {
                            "path": os.path.join(shot_path, f"basecolor{extension}"),
                            "default_color": (1.0, 1.0, 1.0, 1.0),  # white
                            "colorspace": "sRGB",
                        },
                        "INPUT_NORMAL": {
                            "path": os.path.join(shot_path, f"normal{extension}"),
                            "default_color": (
                                0.5,
                                0.5,
                                1.0,
                                1.0,
                            ),  # Default normal map (pointing forward)
                            "colorspace": "Non-Color",
                        },
                        "INPUT_ROUGHNESS": {
                            "path": os.path.join(shot_path, f"roughness{extension}"),
                            "default_color": (0.5, 0.5, 0.5, 1.0),  # Medium roughness
                            "colorspace": "Non-Color",
                        },
                        "INPUT_REFLECTIVITY": {
                            "path": os.path.join(shot_path, f"specular{extension}"),
                            "default_color": (
                                0.5,
                                0.5,
                                0.5,
                                1.0,
                            ),  # Default dielectric specular
                            "colorspace": "Non-Color",
                        },
                        "INPUT_METALLIC": {
                            "path": os.path.join(shot_path, f"metallic{extension}"),
                            "default_color": (0.0, 0.0, 0.0, 1.0),  # Default metallic
                            "colorspace": "Non-Color",
                        },
                        "INPUT_ALPHA": {
                            "path": os.path.join(shot_path, f"alpha{extension}"),
                            "default_color": (1.0, 1.0, 1.0, 1.0),  # Fully opaque
                            "colorspace": "Non-Color",
                        },
                    }

        warnings = []
        nodes = material_obj.node_tree.nodes
        print("nodes in apply_pbr_maps: ")
        for node in nodes:
            print(f"node: {node}")
        if use_video_atlas:
            image_sRGB = bpy.data.images.new(
                f"INPUT_SRGB_map.{suffix}", width=1920, height=1080
            )
            mp4_exist = os.path.exists(os.path.join(shot_path, "output.mp4"))
            png_exist = os.path.exists(os.path.join(shot_path, "output.png"))
            if mp4_exist:
                image_sRGB.filepath = os.path.join(shot_path, "output.mp4")
                image_sRGB.source = "MOVIE"
            elif png_exist:
                image_sRGB.filepath = os.path.join(shot_path, "output.png")
                image_sRGB.source = "FILE"
            else:
                return False, "No video or image found"
            image_sRGB.colorspace_settings.name = "sRGB"

            image_non_color = bpy.data.images.new(
                f"INPUT_NON_COLOR_map.{suffix}", width=1920, height=1080
            )
            if mp4_exist:
                image_non_color.filepath = os.path.join(shot_path, "output.mp4")
                image_non_color.source = "MOVIE"
            elif png_exist:
                image_non_color.filepath = os.path.join(shot_path, "output.png")
                image_non_color.source = "FILE"
            else:
                return False, "No video or image found"
            image_non_color.colorspace_settings.name = "Non-Color"

            labels = [
                "INPUT_VIDEO",
                "INPUT_BASECOLOR",
                "INPUT_NORMAL",
                "INPUT_ROUGHNESS",
                "INPUT_REFLECTIVITY",
                "INPUT_METALLIC",
                "INPUT_ALPHA",
            ]

            for label in labels:
                image = (
                    image_sRGB
                    if label in ["INPUT_VIDEO", "INPUT_BASECOLOR"]
                    else image_non_color
                )

                for node in nodes:
                    if node.type == "GROUP" and node.label == label:
                        for subnode in node.node_tree.nodes:
                            if subnode.type == "TEX_IMAGE":
                                subnode.image = image
                                subnode.image_user.use_auto_refresh = True

                                subnode.image_user.frame_duration = camera_data[
                                    "frame_count"
                                ]
                                subnode.image_user.frame_start = 0
                                subnode.image_user.frame_offset = (
                                    0  # Use 0 for video atlas
                                )

                                print(f"node: {subnode}")
                                print(f"node.image_user: {subnode.image_user}")

        else:
            # Process each map type
            for label, map_info in map_definitions.items():
                use_default = False

                if os.path.exists(map_info["path"]):
                    # Load existing image
                    image = bpy.data.images.load(map_info["path"], check_existing=True)
                    image.name = f"{label}_map.{suffix}"

                    if map_info["path"].endswith(".mp4"):
                        image.source = "MOVIE"

                    if not map_info["path"].endswith(".mp4") and shot_type == "video":
                        image.source = "SEQUENCE"

                    image.colorspace_settings.name = map_info["colorspace"]
                else:
                    use_default = True
                    # Create default texture
                    warnings.append(
                        f"Missing {label} map at {map_info['path']}. Using default values."
                    )

                    # Create a new image with default values
                    image = bpy.data.images.new(
                        name=f"{label}_map.{suffix}",
                        width=1024,
                        height=1024,
                        alpha=True,
                        float_buffer=True,
                    )
                    image.source = "GENERATED"

                    # Set default color
                    image.generated_color = map_info["default_color"]
                    image.colorspace_settings.name = map_info["colorspace"]

                # Update material nodes
                for node in nodes:
                    if node.type == "TEX_IMAGE" and node.label == label:
                        print(f"node: {node}")
                        node.image = image
                        node.image_user.use_auto_refresh = True

                        print(f"shot_type: {shot_type}")
                        if shot_type == "video" and not use_default:
                            node.image_user.frame_duration = camera_data["frame_count"]
                            node.image_user.frame_start = 0
                            if map_info["path"].endswith(".mp4"):
                                node.image_user.frame_offset = 0
                            else:
                                start_num = int(
                                    map_info["path"]
                                    .split("/")[-1]
                                    .split("_")[-1]
                                    .split(".")[0]
                                )
                                if start_num == 1:
                                    node.image_user.frame_offset = 0
                                elif start_num == 0:
                                    node.image_user.frame_offset = -1
                        print(f"node.image_user: {node.image_user}")

        # Return success with warnings if any
        if warnings:
            return True, "PBR maps applied with following warnings:\n" + "\n".join(
                warnings
            )
        return True, "PBR maps applied successfully"

    except Exception as e:
        return False, f"Error applying PBR maps: {str(e)}"


def apply_geometry(
    geometry_modifiers,
    footage,
    shot_type,
    shot_path,
    camera_data,
    suffix,
):
    """
    Apply geometry and depth maps to the geometry modifier object.
    If no position path exists, disables displacement and positions plane at 1m from camera.

    Args:
        geometry_modifiers: Geometry modifier object to apply maps to
        shot_type: Type of shot (video or image)
        shot_path: Path to shot directory
        camera_data: Dictionary containing camera data
        suffix: Suffix for input maps

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        # Set up depth texture dimensions based on orientation
        width = (
            1080
            if camera_data.get("orientation")
            in [ORIENTATION_PORTRAIT, ORIENTATION_UPSIDE_DOWN]
            else 1920
        )
        height = (
            1920
            if camera_data.get("orientation")
            in [ORIENTATION_PORTRAIT, ORIENTATION_UPSIDE_DOWN]
            else 1080
        )

        # Check if position path exists - support both 6-digit and 4-digit formats
        depth_extension = get_frame_format(shot_path, "Depth", "exr")
        depth_path = os.path.join(shot_path, f"Depth{depth_extension}")
        if not os.path.exists(depth_path):
            depth_path = os.path.join(shot_path, "Depth.mp4")
        if not os.path.exists(depth_path):
            depth_extension = get_frame_format(shot_path, "depth", "exr")
            depth_path = os.path.join(shot_path, f"depth{depth_extension}")
        if not os.path.exists(depth_path):
            depth_extension = get_frame_format(shot_path, "depth_norm", "exr")
            depth_path = os.path.join(shot_path, f"depth_norm{depth_extension}")
        if not os.path.exists(depth_path):
            depth_extension = get_frame_format(shot_path, "position", "exr")
            depth_path = os.path.join(shot_path, f"position{depth_extension}")
        depth_exists = os.path.exists(depth_path)

        if depth_exists:
            # Create and configure depth texture image
            depth_texture_image = bpy.data.images.new(
                f"INPUT_DEPTH_map.{suffix}", width=width, height=height
            )

            if shot_type == "video":
                if depth_path.endswith(".mp4"):
                    depth_texture_image.source = "MOVIE"
                else:
                    depth_texture_image.source = "SEQUENCE"
            else:
                depth_texture_image.source = "FILE"

            depth_texture_image.filepath = depth_path
            depth_texture_image.colorspace_settings.name = "Non-Color"
            # Apply displace modifiers with depth map
            geometry_modifiers["displace_modifier"]["Socket_2"] = depth_texture_image
            try:
                geometry_modifiers["displace_modifier"]["Socket_4"] = camera_data.get(
                    "scale"
                )
            except Exception as e:
                return False, f"Error setting scale for geometry modifier: {str(e)}"
        else:
            # Disable displacement and position plane
            geometry_modifiers["displace_modifier"].show_in_editmode = False
            geometry_modifiers["displace_modifier"].show_viewport = False
            geometry_modifiers["displace_modifier"].show_render = False

            # Set plane position 1m away from camera along its local Z-axis
            footage.location = mathutils.Vector((0, 0, -1))

            # Calculate and set plane scale to match camera FOV at 1m distance
            plane_height = (
                2
                * camera_data["sensor_height"]
                / camera_data["frames"][0]["focal_length"]
            )
            plane_width = plane_height * (width / height)  # Maintain aspect ratio

            footage.scale = (plane_width / 4, plane_height / 4, 1)

        # Apply geometry modifiers
        if not geometry_modifiers:
            return False, "No geometry modifier object found"

        ## TODO: Shader node setup하고 중복됨
        # Check if alpha path exists - support both 6-digit and 4-digit formats
        alpha_extension = get_frame_format(shot_path, "alpha", "png")
        alpha_path = os.path.join(shot_path, f"alpha{alpha_extension}")
        alpha_exists = os.path.exists(alpha_path)

        if alpha_exists:
            alpha_texture_image = bpy.data.images.new(
                f"INPUT_GEOMETRY_ALPHA_map.{suffix}", width=width, height=height
            )
            if shot_type == "video":
                alpha_texture_image.source = "SEQUENCE"
            else:
                alpha_texture_image.source = "FILE"
            alpha_texture_image.filepath = alpha_path

        # Apply delete modifiers
        geometry_modifiers["delete_modifier"]["Socket_2"] = alpha_texture_image

        geometry_modifiers.update_tag()
        return True, "Geometry applied successfully"

    except Exception as e:
        return False, f"Error applying geometry: {str(e)}"


def apply_lens_tracking_legacy(scene, lens, camera_data):
    """
    Apply camera animation data to the camera object.

    Args:
        scene: Current Blender scene
        camera_obj: Camera object to apply tracking to
        camera_data: Dictionary containing camera tracking data

    Returns:
        bool: True if successful, False otherwise
    """
    if not lens:
        return False, "No object found"

    try:
        frame_datas = camera_data.get("frames")
        focal_lengths = [frame["focal_length"] for frame in frame_datas]
        sensor_height = camera_data.get("sensor_height")

        # Ensure animation data exists with descriptive names
        if not lens.animation_data:
            lens.animation_data_create()
        if not lens.animation_data.action:
            lens.animation_data.action = bpy.data.actions.new(name=f"Lens_{lens.name}")

        # Create f-curves
        fcurves = {}
        fcurves["sensor_height"] = lens.animation_data.action.fcurves.new(
            data_path="sensor_height"
        )
        set_action_slot_if_compatible(lens.animation_data)

        # Set up camera properties
        lens.sensor_fit = "VERTICAL"
        lens.lens_unit = "MILLIMETERS"
        lens.clip_start = 0.0001
        lens.clip_end = 1000.0

        # Insert keyframe points
        for frameidx, focal_length in enumerate(focal_lengths):
            # Set and keyframe lens and sensor height
            frame_number = frameidx + scene.frame_start
            if focal_lengths[frameidx] != -1:
                lens.sensor_height = sensor_height
                fcurves["sensor_height"].keyframe_points.insert(
                    frame=frame_number, value=lens.sensor_height, options={"FAST"}
                )

        # Update scene and camera object
        lens.update_tag()
        scene.view_layers.update()

        return True, "Camera tracking applied successfully"

    except Exception as e:
        return False, f"Error applying camera tracking: {str(e)}"


def apply_focal_length_tracking_legacy(scene, obj, camera_data, use_iphone_video):
    """
    Apply focal length animation data to the camera anchor object.

    Args:
        scene: Current Blender scene
        obj: Camera anchor object to apply tracking to
        camera_data: Dictionary containing camera tracking data

    Returns:
        bool: True if successful, False otherwise
    """
    if not use_iphone_video:
        return True, "Focal length tracking not required for this video type"

    else:
        if not obj:
            return False, "No object found"

        try:
            frame_datas = camera_data.get("frames")
            focal_lengths = [frame["focal_length"] for frame in frame_datas]

            # Create f-curves
            fcurves = {}
            fcurves["focal_length"] = obj.animation_data.action.fcurves.new(
                data_path='["focal_length"]'
            )
            set_action_slot_if_compatible(obj.animation_data)

            # Insert keyframe points
            for frameidx, focal_length in enumerate(focal_lengths):
                # Set and keyframe focal length
                frame_number = frameidx + scene.frame_start
                if focal_lengths[frameidx] != -1:
                    fcurves["focal_length"].keyframe_points.insert(
                        frame=frame_number, value=focal_length, options={"FAST"}
                    )

            # Update scene and camera object
            obj.update_tag()
            scene.view_layers.update()

            return True, "Camera tracking applied successfully"

        except Exception as e:
            return False, f"Error applying camera tracking: {str(e)}"


def apply_camera_tracking_legacy(scene, obj, camera_data, use_iphone_video):
    """
    Apply camera animation data to the camera object.

    Args:
        scene: Current Blender scene
        camera_obj: Camera object to apply tracking to
        camera_data: Dictionary containing camera tracking data

    Returns:
        bool: True if successful, False otherwise
    """
    if not obj:
        return False, "No object found"

    if not use_iphone_video:
        return True, "Camera tracking not required for this video type"

    try:
        frame_datas = camera_data.get("frames")
        camera_transforms = [frame["transform"] for frame in frame_datas]

        # Validate camera transforms
        if not camera_transforms or not isinstance(camera_transforms, list):
            return False, "Invalid camera transforms"

        # Ensure that each transform is a valid 4x4 matrix
        valid_transforms = []
        for transform in camera_transforms:
            if (
                isinstance(transform, (list, tuple, mathutils.Matrix))
                and len(transform) == 4
                and len(transform[0]) == 4
            ):
                valid_transforms.append(mathutils.Matrix(transform))
        if not valid_transforms:
            return False, "No valid camera transforms found"

        # Clear existing keyframes
        if obj.animation_data and obj.animation_data.action:
            action = obj.animation_data.action
            fcurves_to_clear = [
                "location",
                "rotation_euler",
                "rotation_quaternion",
                "scale",
            ]
            for data_path in fcurves_to_clear:
                fcurves = [fc for fc in action.fcurves if fc.data_path == data_path]
                for fc in fcurves:
                    action.fcurves.remove(fc)

        # Ensure animation data exists with descriptive names
        if not obj.animation_data:
            obj.animation_data_create()
        if not obj.animation_data.action:
            obj.animation_data.action = bpy.data.actions.new(
                name=f"Transform_{obj.name}"
            )

        # Create f-curves
        fcurves = {}
        for data_path in ["location", "rotation_euler", "scale"]:
            fcurves[data_path] = []
            for i in range(3):
                fc = obj.animation_data.action.fcurves.new(data_path=data_path, index=i)
                fcurves[data_path].append(fc)
            set_action_slot_if_compatible(obj.animation_data)

        # Determine if camera was not given, in which case force rotation to 90,0,0 (degrees)
        camera_given_false = False
        try:
            if frame_datas and isinstance(frame_datas, list):
                camera_given_false = frame_datas[0].get("camera_given") is False
        except Exception:
            camera_given_false = False

        forced_rotation = (math.radians(90.0), 0.0, 0.0) if camera_given_false else None

        # Insert keyframe points
        for frameidx, camera_transform in enumerate(valid_transforms):
            frame_number = frameidx + scene.frame_start

            # Calculate relative transform using the dedicated function
            relative_camera_transform = calculate_relative_camera_transform(
                camera_transform, camera_data
            )

            # Set camera transform
            obj.matrix_world = relative_camera_transform

            # Insert keyframes
            for i in range(3):
                fcurves["location"][i].keyframe_points.insert(
                    frame=frame_number, value=obj.location[i], options={"FAST"}
                )
                fcurves["rotation_euler"][i].keyframe_points.insert(
                    frame=frame_number,
                    value=(
                        forced_rotation[i]
                        if forced_rotation is not None
                        else obj.rotation_euler[i]
                    ),
                    options={"FAST"},
                )
                fcurves["scale"][i].keyframe_points.insert(
                    frame=frame_number, value=obj.scale[i], options={"FAST"}
                )

        # Update scene and camera object
        obj.update_tag()
        scene.view_layers.update()

        return True, "Camera tracking applied successfully"

    except Exception as e:
        return False, f"Error applying camera tracking: {str(e)}"


def apply_autofocus_legacy(scene, camera_data, focus_obj, use_iphone_video):
    """
    Apply autofocus animation to the focus object.

    Args:
        scene: Current Blender scene
        camera_data: Dictionary containing camera data
        focus_obj: Focus object to apply animation to

    Returns:
        bool: True if successful, False otherwise
    """
    if not use_iphone_video:
        return True, "Autofocus not required for this video type"

    try:
        if not focus_obj:
            return False, "No focus object selected"

        # Get focus positions
        frame_datas = camera_data.get("frames")
        scale = camera_data.get("scale")
        focus_positions = [
            [
                frame["position_median"][0] * scale,
                frame["position_median"][1] * scale,
                frame["position_median"][2] * scale,
            ]
            for frame in frame_datas
        ]

        # Validate focus_positions
        if not focus_positions or not isinstance(focus_positions, list):
            return False, "Invalid center positions provided for autofocus"

        # Ensure that each center_position is a valid 3D vector
        valid_positions = []
        for pos in focus_positions:
            if isinstance(pos, (list, tuple, mathutils.Vector)) and len(pos) == 3:
                valid_positions.append(mathutils.Vector(pos))

        if not valid_positions:
            return False, "No valid center positions found for autofocus"

        # Clear existing keyframes for 'location'
        if focus_obj.animation_data and focus_obj.animation_data.action:
            action = focus_obj.animation_data.action
            fcurves = [fc for fc in action.fcurves if fc.data_path == "location"]
            for fc in fcurves:
                action.fcurves.remove(fc)

        # Ensure animation data exists
        if not focus_obj.animation_data:
            focus_obj.animation_data_create()

        if not focus_obj.animation_data.action:
            focus_obj.animation_data.action = bpy.data.actions.new(
                name=f"FocusTransform__{focus_obj.name.replace('Focus_', '')}"
            )

        action = focus_obj.animation_data.action

        # Create f-curves for x, y, z location
        fcurves = []
        for i in range(3):
            fc = action.fcurves.new(data_path="location", index=i)
            fcurves.append(fc)
        set_action_slot_if_compatible(focus_obj.animation_data)

        # Insert keyframe points
        for frameidx, center_position in enumerate(valid_positions):
            frame_number = frameidx + scene.frame_start
            for i in range(3):
                fcurves[i].keyframe_points.insert(
                    frame=frame_number, value=center_position[i], options={"FAST"}
                )

        # Update scene and focus object
        focus_obj.update_tag()
        scene.view_layers.update()

        return True, "Autofocus applied successfully"

    except Exception as e:
        return False, f"Error applying autofocus: {str(e)}"


def apply_foot_reference_legacy(
    scene, camera_data, foot_reference_obj, use_iphone_video
):
    """
    Apply autofocus animation to the focus object.

    Args:
        scene: Current Blender scene
        camera_data: Dictionary containing camera data
        foot_reference_obj: Foot reference object to apply animation to

    Returns:
        bool: True if successful, False otherwise
    """
    try:
        if not foot_reference_obj:
            return False, "No focus object selected"

        # Get focus positions
        frame_datas = camera_data.get("frames")
        scale = camera_data.get("scale")

        if not use_iphone_video:
            # For non-iPhone video, just set the location from the first frame without keyframing
            if "position_foot" not in frame_datas[0]:
                first_position = [
                    frame_datas[0]["position_median"][0] * scale,
                    frame_datas[0]["position_median"][1] * scale,
                    frame_datas[0]["position_median"][2] * scale,
                ]
            else:
                first_position = [
                    frame_datas[0]["position_foot"][0] * scale,
                    frame_datas[0]["position_foot"][1] * scale,
                    frame_datas[0]["position_foot"][2] * scale,
                ]

            # Set the location directly without keyframing
            foot_reference_obj.location = mathutils.Vector(first_position)

            # Update scene and focus object
            foot_reference_obj.update_tag()
            scene.view_layers.update()

            return True, "Foot reference position set successfully"

        # For iPhone video, apply keyframe animation
        if "position_foot" not in frame_datas[0]:
            focus_positions = [
                [
                    frame["position_median"][0] * scale,
                    frame["position_median"][1] * scale,
                    frame["position_median"][2] * scale,
                ]
                for frame in frame_datas
            ]
        else:
            focus_positions = [
                [
                    frame["position_foot"][0] * scale,
                    frame["position_foot"][1] * scale,
                    frame["position_foot"][2] * scale,
                ]
                for frame in frame_datas
            ]

        # Validate focus_positions
        if not focus_positions or not isinstance(focus_positions, list):
            return False, "Invalid center positions provided for autofocus"

        # Ensure that each center_position is a valid 3D vector
        valid_positions = []
        for pos in focus_positions:
            if isinstance(pos, (list, tuple, mathutils.Vector)) and len(pos) == 3:
                valid_positions.append(mathutils.Vector(pos))

        if not valid_positions:
            return False, "No valid center positions found for autofocus"

        # Clear existing keyframes for 'location'
        if (
            foot_reference_obj.animation_data
            and foot_reference_obj.animation_data.action
        ):
            action = foot_reference_obj.animation_data.action
            fcurves = [fc for fc in action.fcurves if fc.data_path == "location"]
            for fc in fcurves:
                action.fcurves.remove(fc)

        # Ensure animation data exists
        if not foot_reference_obj.animation_data:
            foot_reference_obj.animation_data_create()

        if not foot_reference_obj.animation_data.action:
            foot_reference_obj.animation_data.action = bpy.data.actions.new(
                name=f"FocusTransform__{foot_reference_obj.name.replace('Focus_', '')}"
            )

        action = foot_reference_obj.animation_data.action

        # Create f-curves for x, y, z location
        fcurves = []
        for i in range(3):
            fc = action.fcurves.new(data_path="location", index=i)
            fcurves.append(fc)
        set_action_slot_if_compatible(foot_reference_obj.animation_data)

        # Insert keyframe points
        for frameidx, center_position in enumerate(valid_positions):
            frame_number = frameidx + scene.frame_start
            for i in range(3):
                fcurves[i].keyframe_points.insert(
                    frame=frame_number, value=center_position[i], options={"FAST"}
                )

        # Update scene and focus object
        foot_reference_obj.update_tag()
        scene.view_layers.update()

        return True, "Autofocus applied successfully"

    except Exception as e:
        return False, f"Error applying autofocus: {str(e)}"


def apply_depth_min_max_legacy(scene, camera_data, obj):
    """
    Apply depth min/max animation data to the camera anchor object.

    Args:
        scene: Current Blender scene
        camera_data: Dictionary containing camera tracking data
        obj: Camera anchor object to apply tracking to

    Returns:
        bool: True if successful, False otherwise
    """
    if not obj:
        return False, "No object found"

    try:
        frame_datas = camera_data.get("frames")
        depth_mins = [frame["depth_min"] for frame in frame_datas]
        depth_maxs = [frame["depth_max"] for frame in frame_datas]

        if not obj.animation_data:
            obj.animation_data_create()
        if not obj.animation_data.action:
            obj.animation_data.action = bpy.data.actions.new(
                name=f"DepthMinMax_{obj.name}"
            )

        # Create f-curves
        fcurves = {}
        fcurves["depth_min"] = obj.animation_data.action.fcurves.new(
            data_path='["depth_min"]'
        )
        fcurves["depth_max"] = obj.animation_data.action.fcurves.new(
            data_path='["depth_max"]'
        )
        set_action_slot_if_compatible(obj.animation_data)

        # Insert keyframe points
        for frameidx, (depth_min, depth_max) in enumerate(zip(depth_mins, depth_maxs)):
            frame_number = frameidx + scene.frame_start

            if depth_mins[frameidx] != -1:
                fcurves["depth_min"].keyframe_points.insert(
                    frame=frame_number, value=depth_min, options={"FAST"}
                )

            if depth_maxs[frameidx] != -1:
                fcurves["depth_max"].keyframe_points.insert(
                    frame=frame_number, value=depth_max, options={"FAST"}
                )

        # Update scene and camera object
        obj.update_tag()
        scene.view_layers.update()

        return True, "Camera tracking applied successfully"

    except Exception as e:
        return False, f"Error applying camera tracking: {str(e)}"


# ------------------------------------------------------------
# Common helpers: legacy (<5.0) + new (>=5.0)
# ------------------------------------------------------------


def is_blender_5_plus():
    return bpy.app.version >= (5, 0, 0)


if is_blender_5_plus():
    from bpy_extras.anim_utils import action_ensure_channelbag_for_slot


def ensure_animdata_and_action(idblock, action_name):
    if idblock.animation_data is None:
        idblock.animation_data_create()
    if idblock.animation_data.action is None:
        idblock.animation_data.action = bpy.data.actions.new(name=action_name)
    return idblock.animation_data.action


def ensure_slot_and_channelbag(action, idblock, slot_name=None):
    """
    Blender 5.0+:
    1) idblock.animation_data.action_slot 이 이미 있고 action이 같고 타입 호환이면 무조건 재사용
    2) 없으면 name/type으로 찾고, 그래도 없으면 새로 생성
    """
    slot_name = slot_name or idblock.name

    # 1) 이미 배정된 slot 재사용 (가장 중요!)
    ad = idblock.animation_data
    if ad and ad.action == action and ad.action_slot:
        s = ad.action_slot
        if s.target_id_type in {"UNSPECIFIED", idblock.id_type}:
            from bpy_extras.anim_utils import action_ensure_channelbag_for_slot

            channelbag = action_ensure_channelbag_for_slot(action, s)
            return s, channelbag

    # 2) action 안에서 name/type으로 찾기
    slot = None
    for s in action.slots:
        if s.name == slot_name and s.target_id_type in {"UNSPECIFIED", idblock.id_type}:
            slot = s
            break

    # 3) 없으면 새로 만들기
    if slot is None:
        slot = action.slots.new(idblock.id_type, slot_name)

    # slot을 평가 대상으로 지정
    if idblock.animation_data is None:
        idblock.animation_data_create()
    idblock.animation_data.action = action
    idblock.animation_data.action_slot = (
        slot  # 5.0+에서 필수 :contentReference[oaicite:1]{index=1}
    )

    from bpy_extras.anim_utils import action_ensure_channelbag_for_slot

    channelbag = action_ensure_channelbag_for_slot(
        action, slot
    )  # layer/strip 자동 생성 :contentReference[oaicite:2]{index=2}

    return slot, channelbag


def get_curve_owner(idblock, action_name, slot_name=None):
    action = ensure_animdata_and_action(idblock, action_name)
    if is_blender_5_plus():
        _, channelbag = ensure_slot_and_channelbag(action, idblock, slot_name=slot_name)
        return action, channelbag
    else:
        return action, action


def clear_fcurves(curve_owner, data_paths):
    for fc in list(curve_owner.fcurves):
        if fc.data_path in data_paths:
            curve_owner.fcurves.remove(fc)


def ensure_fcurve(curve_owner, data_path, index=None):
    fcurves = curve_owner.fcurves
    if is_blender_5_plus():
        if index is None:
            return fcurves.ensure(data_path=data_path)
        return fcurves.ensure(data_path=data_path, index=index)
    else:
        if index is None:
            return fcurves.new(data_path=data_path)
        return fcurves.new(data_path=data_path, index=index)


def apply_lens_tracking(scene, lens, camera_data):
    if not lens:
        return False, "No object found"

    try:
        frame_datas = camera_data.get("frames") or []
        focal_lengths = [frame.get("focal_length", -1) for frame in frame_datas]
        sensor_height = camera_data.get("sensor_height")

        action, curve_owner = get_curve_owner(
            lens, action_name=f"Lens_{lens.name}", slot_name=lens.name
        )

        clear_fcurves(curve_owner, {"sensor_height"})
        f_sensor_h = ensure_fcurve(curve_owner, "sensor_height")

        lens.sensor_fit = "VERTICAL"
        lens.lens_unit = "MILLIMETERS"
        lens.clip_start = 0.0001
        lens.clip_end = 1000.0

        for frameidx, focal_length in enumerate(focal_lengths):
            frame_number = frameidx + scene.frame_start
            if focal_length != -1:
                lens.sensor_height = sensor_height
                f_sensor_h.keyframe_points.insert(
                    frame=frame_number, value=lens.sensor_height, options={"FAST"}
                )

        lens.update_tag()
        scene.view_layers.update()
        return True, "Lens tracking applied successfully"

    except Exception as e:
        return False, f"Error applying lens tracking: {str(e)}"


def apply_focal_length_tracking(scene, obj, camera_data, use_iphone_video):
    if not use_iphone_video:
        return True, "Focal length tracking not required for this video type"

    if not obj:
        return False, "No object found"

    try:
        frame_datas = camera_data.get("frames") or []
        focal_lengths = [frame.get("focal_length", -1) for frame in frame_datas]

        action, curve_owner = get_curve_owner(
            obj,
            action_name=f"Transform_{obj.name}",  # apply_camera_tracking과 동일 이름!
            slot_name=obj.name,  # 동일 slot 재사용
        )

        clear_fcurves(curve_owner, {'["focal_length"]'})
        f_focal = ensure_fcurve(curve_owner, '["focal_length"]')

        for frameidx, focal_length in enumerate(focal_lengths):
            frame_number = frameidx + scene.frame_start
            if focal_length != -1:
                f_focal.keyframe_points.insert(
                    frame=frame_number, value=focal_length, options={"FAST"}
                )

        obj.update_tag()
        scene.view_layers.update()
        return True, "Focal length tracking applied successfully"

    except Exception as e:
        return False, f"Error applying focal length tracking: {str(e)}"


def apply_camera_tracking(scene, obj, camera_data, use_iphone_video):
    if not obj:
        return False, "No object found"

    if not use_iphone_video:
        return True, "Camera tracking not required for this video type"

    try:
        frame_datas = camera_data.get("frames") or []
        camera_transforms = [frame["transform"] for frame in frame_datas]

        if not camera_transforms or not isinstance(camera_transforms, list):
            return False, "Invalid camera transforms"

        valid_transforms = []
        for transform in camera_transforms:
            if (
                isinstance(transform, (list, tuple, mathutils.Matrix))
                and len(transform) == 4
                and len(transform[0]) == 4
            ):
                valid_transforms.append(mathutils.Matrix(transform))
        if not valid_transforms:
            return False, "No valid camera transforms found"

        action, curve_owner = get_curve_owner(
            obj,
            action_name=f"Transform_{obj.name}",  # apply_focal_length_tracking과 동일!
            slot_name=obj.name,  # 동일 slot 재사용
        )

        clear_fcurves(
            curve_owner, {"location", "rotation_euler", "rotation_quaternion", "scale"}
        )

        fcurves = {"location": [], "rotation_euler": [], "scale": []}
        for data_path in ["location", "rotation_euler", "scale"]:
            for i in range(3):
                fcurves[data_path].append(
                    ensure_fcurve(curve_owner, data_path, index=i)
                )

        camera_given_false = False
        try:
            if frame_datas and isinstance(frame_datas, list):
                camera_given_false = frame_datas[0].get("camera_given") is False
        except Exception:
            camera_given_false = False

        forced_rotation = (math.radians(90.0), 0.0, 0.0) if camera_given_false else None

        for frameidx, camera_transform in enumerate(valid_transforms):
            frame_number = frameidx + scene.frame_start
            relative_camera_transform = calculate_relative_camera_transform(
                camera_transform, camera_data
            )
            obj.matrix_world = relative_camera_transform

            for i in range(3):
                fcurves["location"][i].keyframe_points.insert(
                    frame=frame_number, value=obj.location[i], options={"FAST"}
                )
                fcurves["rotation_euler"][i].keyframe_points.insert(
                    frame=frame_number,
                    value=forced_rotation[i]
                    if forced_rotation
                    else obj.rotation_euler[i],
                    options={"FAST"},
                )
                fcurves["scale"][i].keyframe_points.insert(
                    frame=frame_number, value=obj.scale[i], options={"FAST"}
                )

        obj.update_tag()
        scene.view_layers.update()
        return True, "Camera tracking applied successfully"

    except Exception as e:
        return False, f"Error applying camera tracking: {str(e)}"


def apply_autofocus(scene, camera_data, focus_obj, use_iphone_video):
    if not use_iphone_video:
        return True, "Autofocus not required for this video type"

    try:
        if not focus_obj:
            return False, "No focus object selected"

        frame_datas = camera_data.get("frames") or []
        scale = camera_data.get("scale", 1.0)

        focus_positions = [
            [
                frame["position_median"][0] * scale,
                frame["position_median"][1] * scale,
                frame["position_median"][2] * scale,
            ]
            for frame in frame_datas
        ]

        if not focus_positions or not isinstance(focus_positions, list):
            return False, "Invalid center positions provided for autofocus"

        valid_positions = []
        for pos in focus_positions:
            if isinstance(pos, (list, tuple, mathutils.Vector)) and len(pos) == 3:
                valid_positions.append(mathutils.Vector(pos))
        if not valid_positions:
            return False, "No valid center positions found for autofocus"

        action, curve_owner = get_curve_owner(
            focus_obj,
            action_name=f"FocusTransform__{focus_obj.name.replace('Focus_', '')}",
            slot_name=focus_obj.name,
        )

        clear_fcurves(curve_owner, {"location"})
        loc_fcurves = [
            ensure_fcurve(curve_owner, "location", index=i) for i in range(3)
        ]

        for frameidx, center_position in enumerate(valid_positions):
            frame_number = frameidx + scene.frame_start
            for i in range(3):
                loc_fcurves[i].keyframe_points.insert(
                    frame=frame_number, value=center_position[i], options={"FAST"}
                )

        focus_obj.update_tag()
        scene.view_layers.update()
        return True, "Autofocus applied successfully"

    except Exception as e:
        return False, f"Error applying autofocus: {str(e)}"


def apply_foot_reference(scene, camera_data, foot_reference_obj, use_iphone_video):
    try:
        if not foot_reference_obj:
            return False, "No focus object selected"

        frame_datas = camera_data.get("frames") or []
        scale = camera_data.get("scale", 1.0)

        if not use_iphone_video:
            if "position_foot" not in frame_datas[0]:
                first_position = [
                    frame_datas[0]["position_median"][0] * scale,
                    frame_datas[0]["position_median"][1] * scale,
                    frame_datas[0]["position_median"][2] * scale,
                ]
            else:
                first_position = [
                    frame_datas[0]["position_foot"][0] * scale,
                    frame_datas[0]["position_foot"][1] * scale,
                    frame_datas[0]["position_foot"][2] * scale,
                ]

            foot_reference_obj.location = mathutils.Vector(first_position)
            foot_reference_obj.update_tag()
            scene.view_layers.update()
            return True, "Foot reference position set successfully"

        if "position_foot" not in frame_datas[0]:
            focus_positions = [
                [
                    frame["position_median"][0] * scale,
                    frame["position_median"][1] * scale,
                    frame["position_median"][2] * scale,
                ]
                for frame in frame_datas
            ]
        else:
            focus_positions = [
                [
                    frame["position_foot"][0] * scale,
                    frame["position_foot"][1] * scale,
                    frame["position_foot"][2] * scale,
                ]
                for frame in frame_datas
            ]

        if not focus_positions or not isinstance(focus_positions, list):
            return False, "Invalid center positions provided for autofocus"

        valid_positions = []
        for pos in focus_positions:
            if isinstance(pos, (list, tuple, mathutils.Vector)) and len(pos) == 3:
                valid_positions.append(mathutils.Vector(pos))
        if not valid_positions:
            return False, "No valid center positions found for autofocus"

        action, curve_owner = get_curve_owner(
            foot_reference_obj,
            action_name=f"FocusTransform__{foot_reference_obj.name.replace('Focus_', '')}",
            slot_name=foot_reference_obj.name,
        )

        clear_fcurves(curve_owner, {"location"})
        loc_fcurves = [
            ensure_fcurve(curve_owner, "location", index=i) for i in range(3)
        ]

        for frameidx, center_position in enumerate(valid_positions):
            frame_number = frameidx + scene.frame_start
            for i in range(3):
                loc_fcurves[i].keyframe_points.insert(
                    frame=frame_number, value=center_position[i], options={"FAST"}
                )

        foot_reference_obj.update_tag()
        scene.view_layers.update()
        return True, "Foot reference applied successfully"

    except Exception as e:
        return False, f"Error applying foot reference: {str(e)}"


def apply_depth_min_max(scene, camera_data, obj):
    if not obj:
        return False, "No object found"

    try:
        frame_datas = camera_data.get("frames") or []
        depth_mins = [frame.get("depth_min", -1) for frame in frame_datas]
        depth_maxs = [frame.get("depth_max", -1) for frame in frame_datas]

        action, curve_owner = get_curve_owner(
            obj, action_name=f"DepthMinMax_{obj.name}", slot_name=obj.name
        )

        clear_fcurves(curve_owner, {'["depth_min"]', '["depth_max"]'})
        f_depth_min = ensure_fcurve(curve_owner, '["depth_min"]')
        f_depth_max = ensure_fcurve(curve_owner, '["depth_max"]')

        for frameidx, (depth_min, depth_max) in enumerate(zip(depth_mins, depth_maxs)):
            frame_number = frameidx + scene.frame_start
            if depth_min != -1:
                f_depth_min.keyframe_points.insert(
                    frame=frame_number, value=depth_min, options={"FAST"}
                )
            if depth_max != -1:
                f_depth_max.keyframe_points.insert(
                    frame=frame_number, value=depth_max, options={"FAST"}
                )

        obj.update_tag()
        scene.view_layers.update()
        return True, "Depth min/max tracking applied successfully"

    except Exception as e:
        return False, f"Error applying depth min/max tracking: {str(e)}"


def ensure_template_exists(workspace_path):
    """
    Get the bundled template file path.
    The template is now bundled with the addon, no download needed.

    Args:
        workspace_path: Workspace path (kept for backward compatibility, not used)

    Returns:
        str: Path to template.blend if it exists, None otherwise
    """
    template_path = get_template_path()

    if os.path.exists(template_path):
        logger.info(f"Using bundled template: {template_path}")
        return template_path

    logger.error(f"Bundled template not found: {template_path}")
    return None


def append_collection(filepath, auto_remove=False):
    # Get the absolute path
    abs_path = os.path.abspath(filepath)

    # Check if the file exists
    if not os.path.exists(abs_path):
        print(f"File not found: {abs_path}")
        return

    # Append the collection
    with bpy.data.libraries.load(abs_path) as (data_from, data_to):
        # Verify collection exists
        if "BeebleShot.001" not in data_from.collections:
            raise ValueError(f"Collection BeebleShot.001 not found in template file")

        data_to.collections = ["BeebleShot.001"]

    # Get the loaded collection
    collection = data_to.collections[0]

    # Link collection to scene
    bpy.context.scene.collection.children.link(collection)

    # Remove template library reference
    for lib in bpy.data.libraries:
        bpy.data.libraries.remove(lib, do_unlink=True)

    # Clean up unused data
    for datablock in [
        bpy.data.meshes,
        bpy.data.materials,
        bpy.data.armatures,
        bpy.data.actions,
        bpy.data.images,
        bpy.data.textures,
    ]:
        for item in datablock:
            if item.users == 0:
                datablock.remove(item)

    # Remove the .blend file if auto_remove is True
    if auto_remove:
        os.remove(abs_path)
        logger.info(f"Removed temporary file: {abs_path}")
    else:
        logger.info(f"Keeping template file: {abs_path}")

    logger.info(
        f"Collection '{collection.name}' appended successfully to the root hierarchy."
    )
    return collection, collection.name


def find_objects_from_template(collection_name, use_video_atlas, use_position_map):
    """Find and return key objects from the template collection"""
    objs = {
        "global_anchor": None,
        "camera_anchor": None,
        "camera": None,
        "lens": None,
        "focus": None,
        "foot_reference": None,
        "footage_anchor": None,
        "footage": None,
        "material": None,
        "geometry_modifiers": {
            "delete_modifier": None,
            "displce_modifier": None,
        },
    }

    collection = bpy.data.collections.get(collection_name)
    if collection:
        for obj in collection.objects:
            if "BeebleAnchor" in obj.name:
                objs["global_anchor"] = obj
            elif "BeebleCamAnchor" in obj.name:
                objs["camera_anchor"] = obj
            elif "BeebleCam" in obj.name:
                objs["camera"] = obj
                if obj.data:
                    objs["lens"] = obj.data
            elif "BeebleFocus" in obj.name:
                objs["focus"] = obj
            elif "BeebleFootReference" in obj.name:
                objs["foot_reference"] = obj
            elif "BeebleFootageAnchor" in obj.name:
                objs["footage_anchor"] = obj
            elif "BeebleFootage" in obj.name:
                objs["footage"] = obj
                if obj.data.materials:
                    # Remove all material slots that are not the desired material
                    index = 0
                    for i in range(len(obj.data.materials)):
                        if (
                            not use_video_atlas
                            and "BeebleVideoMaterial" in obj.data.materials[i].name
                        ) or (
                            use_video_atlas
                            and "BeebleImageMaterial" in obj.data.materials[i].name
                        ):
                            index = i
                    obj.data.materials.pop(index=index)
                    objs["material"] = obj.data.materials[0]

                    # use_position_map = True -> VertexDisplacement_IMAGE_POSITION 사용
                    # use_position_map = False -> VertexDisplacement_IMAGE 사용
                    used_modifiers = []
                    used_modifiers.append("PlaneSubdivision_IMAGE")
                    if use_position_map:
                        used_modifiers.append("VertexDisplacement_IMAGE_POSITION")
                        objs["geometry_modifiers"]["displace_modifier"] = obj.modifiers[
                            "VertexDisplacement_IMAGE_POSITION"
                        ]
                    else:
                        used_modifiers.append("VertexDisplacement_IMAGE")
                        objs["geometry_modifiers"]["displace_modifier"] = obj.modifiers[
                            "VertexDisplacement_IMAGE"
                        ]

                    for mod in obj.modifiers:
                        if mod.name not in used_modifiers:
                            obj.modifiers.remove(mod)

    return objs


# TODO: Change to much better algorithm
def smooth_motion(transforms, window_size=51, alpha=0.5):
    """
    Smooth camera transforms using a sliding window approach.
    Uses SLERP for rotation averaging and LERP for position averaging.

    Parameters:
        transforms: List of camera transforms
        window_size: Number of frames to consider in the smoothing window (odd number recommended)
        alpha: Smoothing factor (0 = no smoothing, 1 = full smoothing)
    """
    if not transforms:
        return []

    if window_size < 1:
        window_size = 1

    # Make window_size odd to ensure symmetric window
    if window_size % 2 == 0:
        window_size += 1

    half_window = window_size // 2
    smoothed_transforms = []

    # Add first frame unchanged
    first_matrix = transforms[0].get_matrix().transposed()
    smoothed_transforms.append(first_matrix)

    # Process remaining frames
    for i in range(1, len(transforms)):
        # Calculate window boundaries
        start_idx = max(0, i - half_window)
        end_idx = min(len(transforms), i + half_window + 1)

        # Get current transform
        current_mat = transforms[i].get_matrix().transposed()
        current_rot = current_mat.to_quaternion()
        current_pos = current_mat.to_translation()

        # Initialize accumulator for weighted averaging
        avg_rot = current_rot.copy()
        avg_pos = current_pos.copy()
        total_weight = 1.0

        # Process window frames
        for j in range(start_idx, end_idx):
            if j == i:
                continue  # Skip current frame as it's already included

            # Get transform from window
            window_mat = transforms[j].get_matrix().transposed()
            window_rot = window_mat.to_quaternion()
            window_pos = window_mat.to_translation()

            # Calculate weight based on distance from current frame
            weight = 1.0 - (abs(i - j) / window_size)

            # Accumulate weighted rotation (SLERP)
            avg_rot = avg_rot.slerp(window_rot, weight / (total_weight + weight))

            # Accumulate weighted position (LERP)
            avg_pos = avg_pos.lerp(window_pos, weight / (total_weight + weight))

            total_weight += weight

        # Apply smoothing factor
        final_rot = current_rot.slerp(avg_rot, alpha)
        final_pos = current_pos.lerp(avg_pos, alpha)

        # Normalize rotation
        final_rot.normalize()

        # Construct new transform
        new_mat = final_rot.to_matrix().to_4x4()
        new_mat.translation = final_pos
        smoothed_transforms.append(new_mat)

    return smoothed_transforms


def create_camera_action(camera, base_name="CameraStabilization"):
    """Blender 5.0+ create action + slot + fcurves for camera."""

    # Ensure AnimData
    if camera.animation_data is None:
        camera.animation_data_create()

    # New Action
    action = bpy.data.actions.new(name=base_name)

    if bpy.app.version < (5, 0, 0):
        camera.animation_data.action = action
        loc_curves = [
            action.fcurves.new(data_path="location", index=i) for i in range(3)
        ]
        rot_curves = [
            action.fcurves.new(data_path="rotation_euler", index=i) for i in range(3)
        ]
        set_action_slot_if_compatible(camera.animation_data)
        return loc_curves, rot_curves
    else:  # Blender 5.0+
        # Create a slot for this camera object.
        # In 5.0 slots are required; fcurves live per-slot.
        slot = action.slots.new(camera.id_type, camera.name)

        # Assign action + slot to camera
        camera.animation_data.action = action
        camera.animation_data.action_slot = slot

        # Get (or auto-create) the channelbag for this slot.
        # This will also auto-create the action's layer and keyframe strip if missing.
        channelbag = action_ensure_channelbag_for_slot(action, slot)

        # Create FCurves on the channelbag (5.0+ way)
        loc_curves = [
            channelbag.fcurves.new(data_path="location", index=i) for i in range(3)
        ]
        rot_curves = [
            channelbag.fcurves.new(data_path="rotation_euler", index=i)
            for i in range(3)
        ]

        return loc_curves, rot_curves


def calculate_frame_zoom_factor(
    original_transform, smoothed_transform, camera, aspect_ratio, near=0.5, far=10.0
):
    """
    Calculate correct zoom factor by considering field of view and projection.

    Args:
        original_transform: Original camera transform matrix
        smoothed_transform: Smoothed camera transform matrix
        camera: Camera object containing focal length and sensor information
    """

    # Get camera properties
    focal_length = camera.data.lens
    sensor_height = camera.data.sensor_height

    # Calculate field of view
    fov = 2 * math.atan(sensor_height / (2 * focal_length))

    # Calculate corner positions based on FOV
    half_height = math.tan(fov / 2) * near
    half_width = half_height / aspect_ratio

    # Define frustum corners in camera space
    corners = [
        mathutils.Vector((-half_width, -half_height, -near, 1.0)),
        mathutils.Vector((half_width, -half_height, -near, 1.0)),
        mathutils.Vector((-half_width, half_height, -near, 1.0)),
        mathutils.Vector((half_width, half_height, -near, 1.0)),
        mathutils.Vector((-half_width, -half_height, -far, 1.0)),
        mathutils.Vector((half_width, -half_height, -far, 1.0)),
        mathutils.Vector((-half_width, half_height, -far, 1.0)),
        mathutils.Vector((half_width, half_height, -far, 1.0)),
    ]

    # Transform corners from original camera space to world space
    original_corners_world = [
        original_transform.transposed() @ corner for corner in corners
    ]

    # Transform corners from original camera space to smoothed camera space
    smoothed_inv = smoothed_transform.inverted()
    corners_in_smoothed = [smoothed_inv @ corner for corner in original_corners_world]

    # Project points to screen space
    screen_coords = []
    for corner in corners_in_smoothed:
        # Perspective division
        if corner.w != 0:
            screen_x = corner.x / -corner.z
            screen_y = corner.y / -corner.z
            screen_coords.append(mathutils.Vector((screen_x, screen_y)))

    # Find maximum screen-space coordinate
    max_screen_x = max(abs(p.x) for p in screen_coords)
    max_screen_y = max(abs(p.y) for p in screen_coords)

    # Calculate required zoom factor
    zoom_factor_x = max_screen_x / (math.tan(fov / 2) / aspect_ratio)
    zoom_factor_y = max_screen_y / math.tan(fov / 2)
    zoom_factor = max(zoom_factor_x, zoom_factor_y)

    return max(1.0, zoom_factor)


def update_motion_strength(self, context):
    """Update function for motion stabilization with zoom compensation"""
    shot_item = self
    camera = None
    anchor = None

    # If shot is not from beeble app, skip
    if not shot_item.collection.get("shot_with_beeble_app"):
        return

    # Get camera and anchor from shot collection
    if hasattr(shot_item, "collection") and shot_item.collection:
        for obj in shot_item.collection.objects:
            if "BeebleCam." in obj.name:
                camera = obj
            if "BeebleCamAnchor." in obj.name:
                anchor = obj

    if not camera or not anchor:
        return

    # Clear existing camera animation
    if camera.animation_data and camera.animation_data.action:
        bpy.data.actions.remove(camera.animation_data.action)

    # Get frame range from shot item
    frame_start = shot_item.collection.get("frame_start", context.scene.frame_start)
    frame_end = shot_item.collection.get("frame_end", context.scene.frame_end)

    # Get original transforms from camera_transforms property
    original_transforms = shot_item.camera_transforms
    if not original_transforms:
        return

    # Calculate smoothed motion
    smoothed_transforms = smooth_motion(
        original_transforms, alpha=shot_item.motion_strength
    )

    # Calculate maximum required zoom factor across all frames
    max_zoom_factor = 1.0
    # for frame_idx in range(len(original_transforms)):
    #     original_mat = original_transforms[frame_idx].get_matrix()
    #     smoothed_mat = smoothed_transforms[frame_idx]

    #     aspect_ratio_x = int(shot_item.aspect_ratio.split(":")[0])
    #     aspect_ratio_y = int(shot_item.aspect_ratio.split(":")[1])
    #     aspect_ratio = aspect_ratio_y / aspect_ratio_x

    #     near, far = 0.5, 10.0
    #     zoom_factor = calculate_frame_zoom_factor(
    #         original_mat, smoothed_mat, camera, aspect_ratio, near, far
    #     )
    #     # max_zoom_factor = max(max_zoom_factor, zoom_factor)

    # Apply the calculated zoom factor to the anchor
    anchor["zoom_factor_stabilization"] = max_zoom_factor

    # Force driver to update immediately
    anchor.update_tag()
    camera.update_tag()
    camera.data.update_tag()
    context.view_layer.update()

    # Trigger driver update
    camera.data.lens = camera.data.lens

    # Create new animation data
    loc_curves, rot_curves = create_camera_action(camera)

    # Prepare keyframe data
    loc_keyframes = [[] for _ in range(3)]
    rot_keyframes = [[] for _ in range(3)]

    # Apply stabilization for each frame
    frame_count = frame_end - frame_start + 1
    for frame_idx in range(frame_count):
        current_frame = frame_start + frame_idx

        # Get original and smoothed transforms
        original_mat = original_transforms[frame_idx].get_matrix()
        smoothed_mat = smoothed_transforms[frame_idx].transposed()

        # Calculate delta transform
        delta_mat = smoothed_mat @ original_mat.inverted()
        delta_mat = delta_mat.transposed()

        # Extract location and rotation
        loc = delta_mat.to_translation()
        rot = delta_mat.to_euler()

        # Store keyframe data
        for i in range(3):
            loc_keyframes[i].append((current_frame, loc[i]))
            rot_keyframes[i].append((current_frame, rot[i]))

    # Fast keyframe insertion using fcurves
    for i in range(3):
        # Location keyframes
        loc_curves[i].keyframe_points.add(frame_count)
        loc_curves[i].keyframe_points.foreach_set(
            "co", [x for co in loc_keyframes[i] for x in co]
        )

        # Rotation keyframes
        rot_curves[i].keyframe_points.add(frame_count)
        rot_curves[i].keyframe_points.foreach_set(
            "co", [x for co in rot_keyframes[i] for x in co]
        )

    # Update FCurves
    for curves in [loc_curves, rot_curves]:
        for fc in curves:
            fc.update()


def update_aspect_ratio(self, context):
    def update_camera_aspect_ratio(shot_item, context):
        scene = context.scene
        camera = None

        if hasattr(shot_item, "collection") and shot_item.collection:
            for obj in shot_item.collection.objects:
                if "BeebleCam." in obj.name:
                    camera = obj

        if not camera:
            return

        # render = scene.render
        # render.resolution_x = shot_item.resolution_x
        # render.resolution_y = shot_item.resolution_y

        # ratio_W = int(shot_item.aspect_ratio.split(":")[0])
        # ratio_H = int(shot_item.aspect_ratio.split(":")[1])
        # render.resolution_x = round(render.resolution_y * ratio_W / ratio_H)

        # if render.resolution_x % 2 != 0:
        #     render.resolution_x -= 1

    update_camera_aspect_ratio(self, context)

    # NOTE: Update "zoom_factor_stabilization" to change right after aspect ratio change
    update_motion_strength(self, context)


def update_fps(self, context):
    def update_shot_fps(shot_item, context):
        scene = context.scene
        # if shot_item.fps != "custom":
        #     scene.render.fps = int(shot_item.fps)
        # else:
        #     scene.render.fps = shot_item.custom_fps

    update_shot_fps(self, context)


def update_resolution(self, context):
    def update_scene_resolution(shot_item, context):
        scene = context.scene
        render = scene.render

        # if shot_item.resolution_y % 2 != 0:
        #     shot_item.resolution_y -= 1

        # render.resolution_y = shot_item.resolution_y

        # ratio_W = int(shot_item.aspect_ratio.split(":")[0])
        # ratio_H = int(shot_item.aspect_ratio.split(":")[1])
        # render.resolution_x = round(render.resolution_y * ratio_W / ratio_H)

        # if render.resolution_x % 2 != 0:
        #     render.resolution_x -= 1

    update_scene_resolution(self, context)


def update_use_depth_toggle(self, context):
    """
    Update function for 'use_depth' toggle.
    When toggled off, disables depth/displacement and sets foot_reference to fixed position.
    When toggled on, enables depth/displacement.
    """
    scene = context.scene
    shot_item = self

    # Get required objects
    footage = None

    if shot_item.collection:
        for obj in shot_item.collection.objects:
            if "BeebleFootage" in obj.name:
                footage = obj

    # When toggle is OFF - no depth/displacement
    if shot_item.use_depth:
        footage["use_depth"] = True
        if "2d_to_3d" in footage:
            footage["2d_to_3d"] = 1.0
        shot_item.subdivision_level = 7

    if not shot_item.use_depth:
        footage["use_depth"] = False
        if "2d_to_3d" in footage:
            footage["2d_to_3d"] = 0.0
        shot_item.subdivision_level = 0

    # Force updates
    footage.update_tag()
    context.view_layer.update()


def update_normal_strength(self, context):
    scene = context.scene
    shot_item = self

    # Get required objects
    footage = None

    if shot_item.collection:
        for obj in shot_item.collection.objects:
            if "BeebleFootage" in obj.name:
                footage = obj

    # Find normal strength node
    group_nodes = [
        node for node in footage.active_material.node_tree.nodes if node.type == "GROUP"
    ]
    normal_strength_node = next(
        (node for node in group_nodes if node.label == "NORMAL_STRENGTH"), None
    )
    float_curve_node = normal_strength_node.node_tree.nodes["Float Curve"]

    # Update the curve point
    curve = float_curve_node.mapping.curves[0]
    normal_strength = 0.5 * 2 ** (shot_item.normal_strength - 1)
    curve.points[1].location = (0.5, normal_strength)
    float_curve_node.mapping.update()

    # Force more aggressive node tree updates
    if hasattr(normal_strength_node.node_tree, "is_updated"):
        normal_strength_node.node_tree.is_updated = True
    normal_strength_node.node_tree.update_tag()
    footage.active_material.node_tree.update_tag()

    # Force a full material update
    footage.active_material.update_tag()


def update_subdivision_level(self, context):
    """
    Update function for subdivision level.
    Syncs both levels and render_levels of the PlaneSubdivision_IMAGE modifier.
    """
    scene = context.scene
    shot_item = self

    # Get required objects
    footage = None

    if shot_item.collection:
        for obj in shot_item.collection.objects:
            if "BeebleFootage" in obj.name:
                footage = obj

    if footage and "PlaneSubdivision_IMAGE" in footage.modifiers:
        modifier = footage.modifiers["PlaneSubdivision_IMAGE"]

        # Update both levels and render_levels to the same value
        modifier.levels = shot_item.subdivision_level
        modifier.render_levels = shot_item.subdivision_level

        # Force updates
        footage.update_tag()
        context.view_layer.update()
