import os
import json
import time
import webbrowser

from .api import get_api_client
from .constants import AUTH_SIGNOUT_URL
from .logger import logger
from .helpers import get_addon_version
from pathlib import Path

import bpy

class LoginStatus:
    """Enum-like class for login status"""
    PENDING = "pending"
    AUTHENTICATED = "authenticated"
    ERROR = "error"
    INVALID = "invalid"


class AuthManager:
    """A manager for handling server sessions."""
    _instance = None

    def __init__(self):
        if AuthManager._instance is not None:
            raise Exception("AuthManager is a singleton!")

        self._reset_state()

    def _reset_state(self):
        """Reset all instance variables to their initial state"""
        self._api_client = get_api_client()
        self.session_data = None
        self.last_fetch_time = 0
        self._poll_count = 0
        self.user_data = None
        self.refresh_interval = 3600  # 1 hour in seconds
        self._initialized = False

    @classmethod
    def get_instance(cls):
        if cls._instance is None:
            cls._instance = AuthManager()
        return cls._instance

    @classmethod
    def reset_instance(cls):
        """Reset the singleton instance to its initial state"""
        if cls._instance is not None:
            cls._instance._reset_state()
        else:
            cls._instance = AuthManager()
        return cls._instance


    def _update_auth_state(self, context, status_data: dict, is_session_response: bool) -> bool:
        """
        Unified method to update authentication state from either session or browser login
        Returns True if authentication was successful
        """
        settings = context.scene.account_settings

        if not status_data:
            logger.info("No session restored")
            settings.status_message = "Welcome! Please log in to continue."
            settings.login_state = "LOGGED_OUT"
            return False

        if status_data["Status"] == "error":
            logger.info(f"Error in status data: {status_data}")
            message = status_data.get("Message", "Unknown error")
            settings.status_message = f"{message} \nPlease login again."
            settings.login_state = "LOGGED_OUT"
            return False

        if status_data["Status"] == "invalid" and is_session_response:
            logger.info(f"Invalid session detected: {status_data}")
            settings.status_message = "Welcome! Please log in to continue."
            settings.login_state = "LOGGED_OUT"
            return False

        if status_data["Status"] == "pending":
            settings.status_message = "Please complete login in your browser..."
            settings.login_state = "LOGGING_IN"
            return False

        user_data = status_data.get("UserData", {})
        jwt_token = user_data.get("IdToken", "")

        if not jwt_token:
            settings.status_message = "No token received. \nPlease login again."
            settings.login_state = "LOGGED_OUT"
            return False

        self._api_client.set_jwt_token(jwt_token)

        # Update user data
        settings.user_id = user_data.get("UserID", "")
        settings.user_email = user_data.get("Email", "")
        settings.addon_version = get_addon_version()
        settings.status_message = f"Logged in as {user_data.get('Email', '')}"
        settings.workspace_path = os.path.join(str(Path.home()), "Beeble_Workspace")

        blender_token = self._api_client.get_blender_token()
        if not is_session_response and blender_token:
            setting_path = os.path.join(settings.workspace_path, "settings.json")
            with open(setting_path, 'w') as f:
                json.dump({"blender_token": blender_token }, f, indent=4)
                logger.info(f"Settings saved to {setting_path}")

        self.session_data = status_data
        self.last_fetch_time = time.time()

        settings.login_state = "LOGGED_IN"
        settings.active_tab = "SHOTS"

        if settings.login_state == "LOGGED_IN":
            self._api_client.track_event("blender_addon_login")

        return True

    def restore_session(self, context) -> bool:
        """Initialize session from stored tokens"""
        if self._initialized:
            return True

        try:
            session_data = self._api_client.get_session()
            success = self._update_auth_state(context, session_data, is_session_response=True)
            if success:
                self._initialized = True


        except Exception as e:
            logger.error(f"Session initialization error: {str(e)}")
            return False


    def check_browser_login_status(self, context) -> tuple[str, bool]:
        """Check browser login status and update context"""
        try:
            status_data = self._api_client.get_login_status()
            if not status_data:
                return LoginStatus.ERROR, False

            if status_data["Status"] == "pending":
                return LoginStatus.PENDING, False

            success = self._update_auth_state(context, status_data, is_session_response=False)

            return LoginStatus.AUTHENTICATED if success else LoginStatus.ERROR, success

        except Exception as e:
            logger.error(f"Login status check error: {str(e)}")
            return LoginStatus.ERROR, False

    def logout(self, context):
        """Handle user logout"""
        try:
            if not self._api_client.update_logout_status():
                logger.warning("Failed to update logout status")

            webbrowser.open(AUTH_SIGNOUT_URL)

            # Reset state
            self.session_data = None
            self._initialized = False
            self._api_client.clear_tokens()

            # Remove settings file
            setting_path = os.path.join(context.scene.account_settings.workspace_path, "settings.json")
            if os.path.exists(setting_path):
                os.remove(setting_path)

            # Update context
            settings = context.scene.account_settings
            settings.login_state = "LOGGED_OUT"
            settings.status_message = "Logged out successfully"
            settings.user_id = ""
            settings.user_email = ""
            settings.addon_version = ""
            settings.workspace_path = ""

            return True

        except Exception as e:
            logger.error(f"Logout error: {str(e)}")
            return False

    def _handle_invalid_session(self, context):
        """Handle invalid session state"""
        settings = context.scene.account_settings
        settings.login_state = "LOGGED_OUT"
        settings.status_message = "Session expired. Please login again."
        self._api_client.clear_tokens()
        self.session_data = None