import os
import logging
from typing import Optional

class Logger:
    """A wrapper class for Python's logging module with additional functionality.

    Features:
    - Singleton pattern to ensure only one logger instance
    - Configurable verbosity for console output
    - Automatic log directory creation
    - Support for both file and console logging
    """

    _instance = None

    def __new__(cls):
        if cls._instance is None:
            cls._instance = super().__new__(cls)
        return cls._instance

    def __init__(self) -> None:
        # Only initialize if this is the first time
        if not hasattr(self, '_initialized'):
            self._initialized = True
            self._verbose = True
            self._logger = None
            self._log_file = None

    @property
    def verbose(self) -> bool:
        """Get the current verbosity setting."""
        return self._verbose

    @verbose.setter
    def verbose(self, value: bool) -> None:
        """Set the verbosity level."""
        self._verbose = bool(value)

    def setup_logger(self, log_dir: Optional[str] = None) -> Optional[logging.Logger]:
        """Initialize and configure the logger.

        Args:
            log_dir: Directory path where log files will be stored
                    If None, only console logging will be enabled

        Returns:
            logging.Logger: Configured logger instance or None if initialization fails
        """
        # Return existing logger if already initialized
        if self._logger is not None:
            return self._logger

        try:
            # Create logger
            logger = logging.getLogger("Beeble")
            logger.setLevel(logging.INFO)

            # Create formatters
            file_formatter = logging.Formatter(
                '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
            )
            console_formatter = logging.Formatter(
                '%(levelname)s - %(message)s'
            )

            # Always add console handler for verbose mode
            console_handler = logging.StreamHandler()
            console_handler.setFormatter(console_formatter)
            console_handler.setLevel(logging.INFO)
            logger.addHandler(console_handler)

            # Add file handler if log_dir is provided
            if log_dir:
                try:
                    os.makedirs(log_dir, exist_ok=True)
                    self._log_file = os.path.join(log_dir, "Beeble_addon.log")

                    file_handler = logging.FileHandler(self._log_file, mode='a')
                    file_handler.setFormatter(file_formatter)
                    file_handler.setLevel(logging.INFO)
                    logger.addHandler(file_handler)

                    logger.info('=' * 80)
                    logger.info("Logger initialized for Beeble addon.")
                except OSError as e:
                    logger.error(f"Failed to set up file logging: {e}")
                    logger.info("Continuing with console-only logging.")
            else:
                logger.info("Initialized with console-only logging.")

            self._logger = logger
            return logger

        except Exception as e:
            print(f"Failed to initialize logger: {e}")
            return None

    def _log(self, level: str, message: str) -> None:
        """Internal method to handle logging at different levels.

                Args:
            level: The logging level ('info', 'error', or 'warning')
            message: The message to log
        """
        if hasattr(self._logger, level):
            log_func = getattr(self._logger, level)
            log_func(message)

        if self._verbose and not self._log_file:
            print(message)

    def info(self, message: str) -> None:
        """Log an info level message."""
        self._log('info', message)

    def error(self, message: str) -> None:
        """Log an error level message."""
        self._log('error', message)

    def warning(self, message: str) -> None:
        """Log a warning level message."""
        self._log('warning', message)

    def reset(self) -> None:
        """Reset the logger to its initial state."""
        if self._logger:
            for handler in self._logger.handlers[:]:
                handler.close()
                self._logger.removeHandler(handler)
        self._logger = None
        self._log_file = None

# Singleton instance
logger = Logger()