import os
import bpy
import sys
import hashlib
import platform

from datetime import datetime

from .logger import logger
from .. import bl_info


def purge_unused_data():
    """
    Purges all unused data blocks from the Blender file.
    This includes: materials, textures, images, meshes, actions, worlds, and node groups.
    """
    # Purge all unused data blocks
    for collection_name in dir(bpy.data):
        collection = getattr(bpy.data, collection_name)
        if hasattr(collection, "remove"):
            # Check if it's a valid collection
            if hasattr(collection, "users") and hasattr(collection, "is_library_indirect"):
                for item in collection:
                    # Only remove data blocks that aren't used and aren't from linked libraries
                    if item.users == 0 and not item.use_fake_user and not item.is_library_indirect:
                        collection.remove(item)

    # Force garbage collection
    bpy.ops.outliner.orphans_purge(do_recursive=True)

def validate_name(name: str) -> tuple[bool, str]:
    """
    Validates a user's name with the following rules:
    - No leading spaces (but trailing spaces allowed)
    - Can contain numbers
    - Only allows special characters: underscore(_) and hyphen(-)
    - Supports multiple languages (English, Russian, Spanish, Chinese, Korean, etc.)
    - No empty names
    - Length between 2-50 characters
    - Spaces between letters are allowed
    Returns: (is_valid: bool, error_message: str)
    """
    # Check if starts with space
    if name.startswith(" "):
        return False, "Set name cannot start with a space"

    # Get name without trailing spaces for some checks
    name_no_trailing = name.rstrip()

    # Rule 1: Check if name is empty
    if not name_no_trailing:
        return False, "Set name is required"

    # Rule 2: Check minimum and maximum length
    if len(name_no_trailing) < 2:
        return False, "Set name: Min 2 characters"
    if len(name_no_trailing) > 50:
        return False, "Set name: Max 50 characters"

    # Rule 3: Check for special characters (allow Unicode letters, numbers, underscore, hyphen)
    if not all(c.isalnum() or c.isspace() or c in '_-' for c in name):
        return False, "Set name: letters, numbers, _, - only"

    return True, ""


def get_preview_renderer_path():
    # Get the addon directory
    addon_dir = os.path.dirname(os.path.realpath(__file__))
    # Path to executable inside the addon folder
    if sys.platform == "win32":
        exe_path = os.path.join(addon_dir, "beeble.exe")
    elif sys.platform == "darwin":
        exe_path = os.path.join(addon_dir, "beeble")
    else:  # Linux
        exe_path = os.path.join(addon_dir, "beeble")

    logger.info(f"Preview renderer path: {exe_path}")
    return exe_path


def get_addon_version():
    """
    Get the current addon version from bl_info.
    Returns version as a string in format "X.Y.Z"
    """
    try:
        version = bl_info.get('version', (1, 0, 0))
        return '.'.join(str(x) for x in version)
    except:
        return "1.0.0"


def get_device_id():
    """
    Generate a unique device ID based on hardware and OS information.
    Returns a consistent ID for the same device across sessions.
    """
    # Collect system information
    system_info = [
        platform.node(),           # Computer network name
        platform.machine(),        # Machine type
        platform.processor(),      # Processor type
        platform.system(),         # OS name
        str(platform.version()),   # OS version
    ]

    # Add MAC address if available (more reliable on some systems)
    try:
        import uuid
        system_info.append(hex(uuid.getnode()))  # MAC address
    except:
        pass

    # Create a unique string from system info
    unique_string = ':'.join(system_info)

    # Generate a hash that will be consistent for this device
    device_hash = hashlib.sha256(unique_string.encode()).hexdigest()

    # Use first 32 chars as device ID
    return device_hash[:32]


def convert_modified_time(modified_utc: str) -> str:
    """
    Convert UTC timestamp to local time string.
    """
    if not modified_utc:
        return "Unknown"

    try:
        # Convert ISO format UTC time to datetime object
        utc_dt = datetime.fromisoformat(modified_utc.replace('Z', '+00:00'))

        # Convert to local timezone
        local_dt = utc_dt.astimezone()

        # Format for display
        return local_dt.strftime("%Y-%m-%d %H:%M:%S")

    except (ValueError, AttributeError):
        return "Unknown"


