import logging
logging.getLogger('urllib3').setLevel(logging.WARNING)

import os
import sys
import bpy
import json
import time
import platform
import requests
import mimetypes
from typing import Dict, Optional, List, Tuple
from concurrent.futures import ThreadPoolExecutor

from .logger import logger
from .helpers import get_device_id, get_addon_version
from .constants import API_URL, UNK_ERROR_MSG

class BeebleApiClient:
    _instance = None

    def __new__(cls):
        if cls._instance is None:
            cls._instance = super().__new__(cls)
            cls._instance._initialized = False
        return cls._instance

    def __init__(self, base_url: str = API_URL):
        if not self._initialized:
            self.base_url = base_url.rstrip('/')
            self.session = requests.Session()
            self._initialized = True
            self._token = None
            self._jwt_token = None

    @property
    def headers(self) -> Dict[str, str]:
        """Get headers with JWT token if available"""
        if self._jwt_token:
            return {'Authorization': f'Bearer {self._jwt_token}'}
        return {}

    def set_blender_token(self, token: str):
        """Set blender token"""
        self._token = token

    def get_blender_token(self) -> Optional[str]:
        """Get blender token"""
        return self._token


    def clear_tokens(self):
        """Clear both JWT and refresh tokens"""
        self._jwt_token = None
        self._token = None
        self.user_data = None


    def set_jwt_token(self, jwt_token: str):
        """Set jwt token """
        self._jwt_token = jwt_token


    def refresh_jwt_token(self) -> bool:
        """
        Attempt to get a new JWT token using the refresh token
        Returns: bool indicating success
        """
        try:
            response = requests.post(
                f"{self.base_url}/auth/refresh-token",
                json={'Token': self._token},
                headers={'Content-Type': 'application/json'}
            )
            response.raise_for_status()
            data = response.json()
            user_data = data.get("UserData", {})
            self._jwt_token = user_data.get("IdToken", "")

            return bool(self._jwt_token)

        except Exception as e:
            logger.error(f"Error refreshing JWT token: {str(e)}")
            self.clear_tokens()
            return False


    def _handle_auth_error(self, response) -> bool:
        """
        Handle authentication errors and attempt recovery
        Returns: bool indicating if retry should be attempted
        """
        if response.status_code == 401:
            # Try refreshing the token
            if self.refresh_jwt_token():
                return True  # Retry the request
            else:
                self.clear_tokens()
        return False


    def _make_authenticated_request(self, method: str, endpoint: str, **kwargs) -> Tuple[Optional[Dict], bool]:
        """
        Make an authenticated request with automatic token refresh
        """
        url = f"{self.base_url}/{endpoint.lstrip('/')}"

        # Add JWT token to headers
        request_headers = {}
        if self._jwt_token:
            request_headers['Authorization'] = f'Bearer {self._jwt_token}'

        if 'json' in kwargs and 'files' not in kwargs:
            request_headers['Content-Type'] = 'application/json'

        # Add headers to the request
        kwargs['headers'] = {**kwargs.get('headers', {}), **request_headers}

        for attempt in range(2):
            try:
                response = requests.request(method, url, **kwargs)

                if response.ok:
                    return response.json(), True

                # Handle authentication errors
                if response.status_code == 401:
                    if attempt < 1 and self._handle_auth_error(response):
                        kwargs['headers']['Authorization'] = f'Bearer {self._jwt_token}'
                        continue  # Retry the request
                    logger.error("Authentication failed")
                    return {'error': 'Authentication failed'}, False
                try:
                    error_details = response.json()
                    error_msg = error_details.get('detail', UNK_ERROR_MSG)
                    logger.error(f"Error response: {error_msg}")

                except ValueError:
                    pass

                logger.error(f"Request failed with status {response.status_code}")
                return {'error': error_msg}, False

            except requests.exceptions.RequestException as e:
                logger.error(f"Network error: {str(e)}")
                return {'error': f'Network error: {str(e)}'}, False

            except Exception as e:
                logger.error(f"Unexpected error: {str(e)}")
                return {'error': f'Unexpected error: {str(e)}'}, False


    def issue_blender_token(self) -> Optional[str]:
        """Get blender token"""
        device_id = get_device_id()
        addon_version = get_addon_version()
        os_version = sys.platform

        # Print version
        blender_version = bpy.app.version
        logger.info(f"Blender Version: {blender_version[0]}.{blender_version[1]}.{blender_version[2]}")

        init_data = {
            "DeviceID": device_id,
            "AddonVersion": get_addon_version()
        }

        response_data, success = self._make_authenticated_request(
            'POST',
            'auth/init-login',
            json=init_data
        )

        logger.info(f"Device ID: {device_id}, Addon Version: {addon_version}, OS Version: {os_version}")

        return response_data.get('Token') if success else None

    def get_session(self) -> Optional[str]:

        # Check if we have a token
        if not hasattr(bpy.context.scene, "account_settings") or not bpy.context.scene.account_settings.workspace_path:
            logger.error("Setting not accessible")
            return None

        # Read setting.json file
        setting_path = os.path.join(bpy.context.scene.account_settings.workspace_path, "settings.json")
        if not os.path.exists(setting_path):
            logger.info("Settings file not found")
            return None

        with open(setting_path, 'r') as f:
            settings = json.load(f)

        # Check if we have a token
        if 'blender_token' not in settings or not settings['blender_token']:
            logger.error("No token available for session request")
            return None

        self._token = settings['blender_token']

        request_data = {
            "Token": self._token,
            "DeviceID": get_device_id(),
            "AddonVersion": get_addon_version()
        }

        response_data, success = self._make_authenticated_request(
            'POST',
            'auth/session',
            json=request_data,
            timeout=5
        )

        if not success:
            logger.error(response_data)

        return response_data if success else None

    def get_login_status(self) -> Optional[Dict]:
        if not self._token:
            return None

        status_data, success = self._make_authenticated_request(
            'GET',
            f'auth/status/{self._token}'
        )

        return status_data if success else None

    def cancel_login(self) -> bool:
        if not self._token:
            return False

        _, success = self._make_authenticated_request(
            'POST',
            f'auth/cancel-login',
            json={'Token': self._token}
        )

        return success


    def update_logout_status(self) -> bool:
        """Update DynamoDB status on logout"""
        if not self._token:
            logger.warning("No refresh token available for logout")
            return False

        _, success = self._make_authenticated_request(
            'POST',
            'auth/logout',
            json={
                'Token': self._token
            }
        )
        return success

    def fetch_set_list(self) -> List[Dict]:
        """Fetch list of all available sets"""
        data, success = self._make_authenticated_request('GET', 'set/list?include_thumbnails=true')
        return data if success else []

    def get_upload_urls(self, set_id: str) -> Optional[Dict]:
        """Initialize a new set and get upload URLs"""
        response_data, success = self._make_authenticated_request(
            'GET',
            f'set/upload-urls/{set_id}'
        )
        if not success:
            return None
        return response_data

    def upload_set_files(self, upload_urls: Dict, files: Dict[str, str]) -> bool:
        """Upload set files concurrently using presigned URLs"""
        def upload_worker(args):
            file_type, file_path = args
            url_type = f'{file_type}_url'
            if url_type not in upload_urls:
                return True
            start_time = time.time()
            result = self.upload_file(upload_urls[url_type], file_path, file_type)
            end_time = time.time()
            logger.info(f"Uploaded {file_type} in {end_time - start_time:.2f} seconds")
            return result

        with ThreadPoolExecutor() as executor:
            results = list(executor.map(upload_worker, files.items()))

        return all(results)

    def upload_file(self, url: str, file_path: str, file_type: str) -> bool:
        """Upload a single file using a presigned URL"""
        try:
            if not file_path and file_type in ['hdri', 'license']:
                return True

            if not os.path.exists(file_path):
                logger.error(f"File not found: {file_path}")
                return False

            content_type = mimetypes.guess_type(file_path)[0] or 'application/octet-stream'
            headers = {'Content-Type': content_type}

            with open(file_path, 'rb') as f:
                response = requests.put(url, data=f, headers=headers)
                response.raise_for_status()
                return True

        except Exception as e:
            logger.error(f"Error uploading file {file_path}: {str(e)}")
            return False


    def upsert_set(self, set_id: str, data: Dict) -> Optional[Dict]:
        """Update set metadata"""
        response_data, success = self._make_authenticated_request(
            'PUT',
            f'set/{set_id}',
            json=data
        )
        return response_data if success else None


    def fetch_asset_list(self, asset_type: str) -> List[Dict]:
        """Fetch list of all available assets

        Args:
            asset_type (str): Type of assets to fetch ("all", "world", "object", etc.)

        Returns:
            List[Dict]: List of asset data dictionaries. Empty list if request fails.
        """
        # Map "world" to "hdri" for API compatibility
        if asset_type == "world":
            api_type = "hdri"
        else:
            api_type = asset_type

        # Build the appropriate endpoint
        if api_type == "all":
            endpoint = 'asset/list'
        else:
            endpoint = f'asset/list?asset_type={api_type}'

        # Make the request
        response_data, success = self._make_authenticated_request('GET', endpoint)
        return response_data if success else []

    def get_asset_details(self, asset_uid: str) -> Optional[Dict]:
        """Get asset details"""
        response_data, success = self._make_authenticated_request(
            'GET',
            f'asset/{asset_uid}'
        )
        return response_data if success else None

    #MARK: - event tracking for analytics (amplitude)
    def track_event(self, event_type: str, event_properties: Optional[dict] = None) -> Tuple[Optional[Dict], bool]:
        """
        Track events using the Amplitude tracking endpoint

        Args:
            event_type: Type of event to track
            event_properties: Optional properties for the event

        Returns:
            Tuple containing response data and success boolean
        """
        try:
            device_id = get_device_id()
            blender_version= f"Blender{bpy.app.version[0]}.{bpy.app.version[1]}.{bpy.app.version[2]}"

            # Prepare tracking data
            track_data = {
                "event_type": event_type,
                "device_id": device_id,
                "os": platform.platform(),
                "blender_version": blender_version,
                "addon_version": get_addon_version(),
                "event_properties": event_properties or {}
            }

            # Make authenticated request to tracking endpoint
            return self._make_authenticated_request(
                'POST',
                'auth/track-event-blender',
                json=track_data
            )

        except Exception as e:
            logger.error(f"Error tracking event: {str(e)}")
            return {'error': f'Error tracking event: {str(e)}'}, False


# Module-level function to get the singleton instance
def get_api_client():
    return BeebleApiClient()