bl_info = {
    "name": "Beeble",
    "author": "Beeble AI Inc.",
    "version": (1, 10, 9),
    "blender": (4, 2, 0),
    "location": "View3D > Sidebar > Beeble",
    "description": "Beeble Blender Addon",
    "category": "3D View",
    # Custom fields for CI/CD (Blender ignores these)
    "release_notes": "Performance improvements",
    "min_supported_version": "1.0.1",
}

import os
import bpy
import shutil
import tempfile
from bpy.app.handlers import persistent

from .interface import register as register_interface
from .interface import unregister as unregister_interface
from .core import register as register_core
from .core import unregister as unregister_core
from .core.logger import logger
from .core.icon_manager import icon_manager
from .core.session import AuthManager
from .core.version_manager import check_for_updates


@persistent
def load_handler(dummy):
    if hasattr(bpy.context.scene, "account_settings"):
        try:
            start_logging(bpy.context)
            get_session(bpy.context)
            remove_temp_directories(bpy.context)
            check_update_notes(bpy.context)

        except Exception as e:
            logger.warning(f"Error during load handler logout: {str(e)}")


def start_logging(context):
    """Start logging"""
    try:
        logger.reset()
        if (
            not hasattr(context.scene, "account_settings")
            or not context.scene.account_settings.workspace_path
        ):
            return
        logger.setup_logger(context.scene.account_settings.workspace_path)

    except Exception as e:
        print(f"Error in start_logging: {str(e)}")


def get_session(context):
    """Get the current session"""
    try:
        AuthManager.reset_instance()
        auth_manager = AuthManager.get_instance()
        auth_manager.restore_session(context)

    except Exception as e:
        print(f"Error in get_session: {str(e)}")


def remove_temp_directories(context):
    """Remove temporary directories except template.blend and version.json"""
    try:
        if (
            not hasattr(context.scene, "account_settings")
            or not context.scene.account_settings.workspace_path
        ):
            print("No workspace path found.")
            return False

        workspace_path = bpy.path.abspath(context.scene.account_settings.workspace_path)
        if not workspace_path:
            print("No workspace path found.")
            return False

        # Define the temp directory
        temp_dir = os.path.join(workspace_path, "Temp")
        if not os.path.exists(temp_dir):
            return True

        # Define the template directory path and files to keep
        templates_dir = os.path.join(temp_dir, "Templates")
        template_path = os.path.join(templates_dir, "template.blend")
        version_path = os.path.join(templates_dir, "version.json")

        # Save template and version files to a temporary location if they exist
        template_backup = None
        version_backup = None

        if os.path.exists(template_path):
            template_backup = tempfile.NamedTemporaryFile(delete=False)
            shutil.copy2(template_path, template_backup.name)

        if os.path.exists(version_path):
            version_backup = tempfile.NamedTemporaryFile(delete=False)
            shutil.copy2(version_path, version_backup.name)

        # Clean up temp directory
        shutil.rmtree(temp_dir)

        # Restore template and version files if they existed
        if template_backup or version_backup:
            os.makedirs(templates_dir, exist_ok=True)

            if template_backup:
                shutil.copy2(template_backup.name, template_path)
                os.unlink(template_backup.name)

            if version_backup:
                shutil.copy2(version_backup.name, version_path)
                os.unlink(version_backup.name)

        print("Temp directories removed (preserved template files).")
        return True

    except Exception as e:
        print(f"Error in remove_temp_directories: {str(e)}")
        return False


def initialize_addon():
    """Initialize the addon"""
    print("Initializing Beeble addon...")
    if bpy.context.window_manager.windows:
        window = bpy.context.window_manager.windows[0]
        screen = window.screen
        area = next((area for area in screen.areas if area.type == "VIEW_3D"), None)

        if area:
            with bpy.context.temp_override(window=window, area=area):
                # Set viewport shading settings
                for space in area.spaces:
                    if space.type == "VIEW_3D":
                        space.shading.use_scene_world = True
                        space.shading.studiolight_intensity = 0.0

    return None


# Register the handlers at module level
if load_handler not in bpy.app.handlers.load_post:
    bpy.app.handlers.load_post.append(load_handler)


def register():
    """Register the addon"""
    register_core()
    register_interface()
    bpy.app.timers.register(initialize_addon, first_interval=0.1)
    icons_dir = os.path.join(os.path.dirname(__file__), "icons")
    icon_manager.load_icons(icons_dir)

    # Don't access bpy.context.scene directly during registration


def unregister():
    """Unregister the addon"""
    # Remove handlers if present
    if load_handler in bpy.app.handlers.load_post:
        bpy.app.handlers.load_post.remove(load_handler)

    # Unregister components
    unregister_interface()
    unregister_core()
    icon_manager.clear_icons()


def check_update_notes(context):
    """Check for update notes and setup update status in context"""
    try:
        # Make sure context and scene are available
        if not context or not hasattr(context, "scene") or not context.scene:
            print("Context or scene not available, skipping update check")
            return

        # Make sure account_settings is available
        if not hasattr(context.scene, "account_settings"):
            print("Account settings not available, skipping update check")
            return

        # Check for updates (no workspace_path needed - fetches from server directly)
        update_needed, update_data = check_for_updates()

        if update_data is None:
            # Reset update status if we couldn't get update data
            context.scene.account_settings.show_update_message = False
            context.scene.account_settings.update_message = ""
            context.scene.account_settings.update_available = False
            context.scene.account_settings.block_addon = False
            return

        # Get latest version from update data (latest.json uses "version" field)
        latest_version = update_data.get("version", "0.0.0")

        # Set block_addon status based on minimum supported version check
        # This was already calculated in check_for_updates based on min_supported_version
        context.scene.account_settings.block_addon = update_data.get(
            "block_addon", False
        )

        # Create update message
        if update_needed:
            if context.scene.account_settings.block_addon:
                default_message = "This version of the addon is not supported. \nPlease update to the latest version."
            else:
                default_message = ""
            message = update_data.get("message", default_message)
            if not message:
                message = default_message

            # Store latest version
            context.scene.account_settings.latest_addon_version = latest_version
        else:
            message = ""

        # Set update message visibility and content
        context.scene.account_settings.show_update_message = update_needed
        context.scene.account_settings.update_message = message
        context.scene.account_settings.update_available = update_needed

    except Exception as e:
        print(f"Error checking update notes: {str(e)}")
        if hasattr(context, "scene") and hasattr(context.scene, "account_settings"):
            context.scene.account_settings.show_update_message = False
            context.scene.account_settings.update_message = ""
            context.scene.account_settings.block_addon = False
            context.scene.account_settings.update_available = False


if __name__ == "__main__":
    register()
